#! /usr/bin/perl

# CAVEAT
# As of October 2025, I haven't used nor otherwise tested this script in well
# over a decade, and I wrote it even earlier. I have no idea whether or not it
# still works with current versions of Perl and MediaWiki.
#
# DESCRIPTION
# This script uploads to a MediaWiki all images in a specified directory with
# a specific extension e.g. all png files.  It will submit an image as a new
# version if the image already exists on the wiki, without checking for changes.
#
# QUICKSTART
# Set the variables below.  The essential ones are $baseurl, $wikiuser, $wikipw,
# $imgdir and $imgext.  Make sure that Perl is installed as well as the
# libwww-perl package on which this script depends.  Then run the script as:
# perl upload.pl
#
# NOTES
# * The string 'Special:Upload' is hard-coded; if the wiki's primary language is
#   other than English, 'Special' might need to be translated in this string.
# * Little parsing of returned HTML pages for error messages is performed -
#   that's left up to the person running the script (set $browser below) -
#   error handling's generally incomplete.
# * Temporary files are deleted only if $browser is set, unless you change
#   $deltmpfilesonexit below.

use Cwd 'abs_path';

## BEGIN CONFIGURABLE VARIABLES
my $baseurl = 'http://domain.of.your.wiki/mediawiki/index.php'; # Must end in
                                                                # index.php
my $wikiuser = 'Yourusername'; # Must have the upload right
my $wikipw = 'yourpassword';
my $imgdir =
  '/absolute/path/to/your/local/directory/containing/images/for/upload';
my $imgext = 'png';     # Case sensitive if your filesystem is too.
my $verbose = 1;        # If 1, show extra information
my $pauseperpage = 0;   # If 1, will pause after each page sent to $browser
                        # until the enter key is pressed.
my $browser = '';       # Path to a browser.  If set, post-submit pages
                        # retrieved from the wiki will be opened in this browser
                        # to allow you to view any success/error messages.
                        # Should be a browser that opens in a separate window;
                        # preferably where separate calls don't invoke a new
                        # browser/window.
my $browserpause = 4;	# Number of seconds to pause to give the browser
                        # time to catch up without skipping pages and prior to
                        # the final tempfile deletion (inapplicable if
                        # $pauseperpage is 1).
my $deltmpfilesonexit = $browser?1:0; # If 1, will delete temporary files on
                                   # exit. Unless you change the default, no
                                   # deletion will be performed when $browser
                                   # is not set so that there's a record for
                                   # review of html error/success messages in
                                   # the files.
my $tmpdir = abs_path("/tmp/wiki_upload_img");
my $htmldir = "$tmpdir/html";
my $cookiefile = "$tmpdir/lwpcookies.txt";
my $tmpfile = "$tmpdir/tmp.html";
## END CONFIGURABLE VARIABLES

use HTTP::Request::Common qw(POST GET);
use LWP::UserAgent;
use HTTP::Cookies;
use File::Copy;

# Init
chdir($imgdir) || die "Cannot chdir to $imgdir: $!";
print STDERR "Logging to stderr\n";
print STDERR "Asterisked items match a page ".
  ($browser ? "opened in \"$browser\"" :
   "that would have been opened in a browser if \$browser had been set")."\n";
mkdir $tmpdir;
$cookiefile = abs_path($cookiefile);
$tmpfile = abs_path($tmpfile);
my @tmpfiles = ($cookiefile, $tmpfile);

# Clean up temp files on ctrl-C
$SIG{INT} = sub {
	if ($deltmpfilesonexit) {
		print STDERR "Removing temporary files.\n";
		&rmtmpfiles;
	}
	exit 1;
};

# Create the HTTP user agent with cookie support
my $ua = LWP::UserAgent->new;
$ua->agent("MyApp/0.1 ");
$ua->cookie_jar(HTTP::Cookies->new(file => $cookiefile, autosave => 1));
mkdir $htmldir;

# Login to wiki
my $req = POST "$baseurl?title=Special:Userlogin&action=submitlogin", [
	wpName => $wikiuser,
	wpPassword => $wikipw,
	wpLoginattempt => 'Log in'
];
my $rc = &send($req);
my $browserfile = "$htmldir/loggedin.html";
push(@tmpfiles, $browserfile);
move($tmpfile, $browserfile);
if (length($browser) > 0) {
	system("$browser $htmldir/loggedin.html &");
}
if (!$rc) {
	print STDERR $res->status_line, "\n";
	print STDERR "*Unable to login to wiki.\n";
	print STDERR "Exiting.\n";
	exit 1;
}
print STDERR "*Logged in to wiki\n";
if ($pauseperpage) {
	&dopause;
} elsif (length($browser) > 0) {
	sleep($browserpause);
}

# Loop over images in directory and upload each one
while (defined($filename = <$imgdir/*.$imgext>)) {
	$dst = $filename;
	$dst =~ s/.*\/(.*)/$1/;
	$verbose && print STDERR "Uploading $filename as $dst\n";
	$url = "$baseurl/Special:Upload";
	$req = POST $url,
		Content_Type => 'form-data',
		Content      => [ wpUploadFile => [$filename],
				  wpDestFile => $dst,
				  wpUploadDescription => '',
				  wpWatchthis => 0,
				  wpUpload => 'Upload file' ];
	if (!&send($req)) {
		print STDERR "HTTP ERROR whilst uploading \"$filename\": ";
		print STDERR $res->status_line, "\n";
		next;
	} elsif ($res->content =~ /form id='uploadwarning'/gs) {
		$verbose && print STDERR "Sending confirmation form to ".
		  "overwrite existing wiki image \"$dst\"\n";
		($wpSessionKey) = $res->content=~/wpSessionKey' value="(\d*)"/;
		$req = POST $url,
			[ wpUploadFile => [$filename],
			  wpIgnoreWarning => 1,
			  wpSessionKey => $wpSessionKey,
			  wpDestFile => $dst,
			  wpUploadDescription => '',
			  wpWatchthis => 0,
			  wpUpload => 'Save file'
			];
		if (!&send($req)) {
			print STDERR "HTTP ERROR whilst confirming upload of ".
			  "\"$filename\": $res->status_line\n";
			next;
		}
	}
	print STDERR "*Uploaded \"$filename\"\n";
	my $savefile = "$htmldir/$dst.uploaded.html";
	$savefile =~ s/ /_/g;
	push(@tmpfiles, $savefile);
	move($tmpfile, $savefile);
	# Display in browser
	if (length($browser) > 0) {
		system("$browser $savefile &");
	}
	if ($pauseperpage) {
		&dopause;
	} elsif (length($browser) > 0) {
		sleep($browserpause);
	}
}

# Delete temp files if specified
if ($deltmpfilesonexit) {
	print STDERR "Removing temporary files.\n";
	&rmtmpfiles;
}

sub send {
	# Pass request to the user agent and get a response back
	my $req = @_[0];
	my $i = 0;
	$res = $ua->request($req);
	if ($res->is_success) {
		open(FILE, ">$tmpfile");
		print FILE $res->content;
		close(FILE); 
		return 1;
	} elsif ($res->status_line == "302 Found") {
		# Temporary redirection - try again
		$req->uri($res->headers->header("Location"));
		$res = $ua->request($req);
		if ($res->is_success) {
			open(FILE, ">$tmpfile");
			print FILE $res->content;
			close(FILE);
			return 1;
		} 
	}
	return 0;
}

sub dopause {
	print "Paused; press enter.";
	my $tmp = <STDIN>;
}

sub rmtmpfiles {
	foreach $filename (@tmpfiles) {
		$verbose && print STDERR "Attempting to delete: '$filename'\n";
		unlink $filename;
	}
	# might not succeed if directories are non-empty
	$verbose && print STDERR "Attempting to remove: '$htmldir'.\n";
	rmdir $htmldir;
	$verbose && print STDERR "Attempting to remove: '$tmpdir'.\n";
	rmdir $tmpdir;
}
