<?php
/**
 * Conditions.php
 *
 * Part of the Hierarchy extension for MediaWiki; supporting the Treeview skin.
 *
 * Licenced under the General Public Licence 2, without warranty.
 *
 * @licence GPL2 http://www.gnu.org/copyleft/gpl.html
 * @author Laird Shaw <lairdshaw77@gmail.com>
 */

if (!defined('MEDIAWIKI')) die();

use MediaWiki\MediaWikiServices;

/** For TvQueryString::queryStrToArr(). */
require_once('QueryString.php');

/* When adding to this list, do not use zero or reuse old numbers - the 2nd is
 * due to the possibility of these constants being used in cached content (not
 * currently implemented) i.e. insert at the end only, and if removing from the
 * end, leave a comment indicating the number that was removed and that it
 * should not be reused.
 */
define('CD_IFLOGGEDIN'              ,  1);
define('CD_IFHASRIGHT'              ,  2);
define('CD_IFISARTICLE'             ,  3);
define('CD_IFISTALK'                ,  4);
define('CD_IFISSPECIAL'             ,  5);
define('CD_IFPAGEEXISTS'            ,  6);
define('CD_IFARTICLEEXISTS'         ,  7);
define('CD_IFTALKEXISTS'            ,  8);
define('CD_IFISFOREIGNUSERPAGE'     ,  9);
define('CD_IFISEMAILABLEUSERPAGE'   , 10);
define('CD_IFVIEWINGOLDTALKPAGE'    , 11);
define('CD_IFVIEWINGOLDARTICLEPAGE' , 12);
define('CD_IFPAGEISDELETED'         , 13);
define('CD_IFARTICLEPAGEISDELETED'  , 14);
define('CD_IFTALKPAGEISDELETED'     , 15);
define('CD_IFARTICLEPAGEISPROTECTED', 16);
define('CD_IFTALKPAGEISPROTECTED'   , 17);
define('CD_IFWATCHINGPAGE'          , 18);
define('CD_IFREDIRECTED'            , 19);
define('CD_IFISARTICLE_PREDIR'      , 20);
define('CD_IFISTALK_PREDIR'         , 21);
define('CD_IFARTICLEEXISTS_PREDIR'  , 22);
define('CD_IFPAGEISDELETED_PREDIR'  , 23);
define('CD_IFTALKEXISTS_PREDIR'     , 24);
define('CD_ALWAYS'                  , 25);
define('CD_IFNODEIDMATCHES'         , 26);
define('CD_IFTITLEMATCHES'          , 27);
define('CD_IFURLEXTRAMATCHES'       , 28);
define('CD_IFURLCONTAINSEXTRA'      , 29);
define('CD_IFURLQUERYCONTAINS'      , 30);
define('CD_IFTITLEMATCHES_POSTREDIR', 31);

/** This class tests various conditions pertaining to a particular web request
  * as defined by a TVstate object.
  *
  * The class is used by the Hierarchy class when interpreting the
  * user-supplied config page defined by $wgHierarchyPage.
  *
  * Some functions are statically callable.
  *
  * @package MediaWiki
  * @subpackage Extensions
  */
class Conditions {
	var $mState;
	/** See getCondition() for a description of array members.
	  * Defaults are:
	  * [0] => required; no default
	  * [1] => false
	  * [2] => false
	  * [3] => $this->mDefaultStartDelimiters
	  * [4] => $this->mDefaultEndDelimiters
	  * [5] => true
	  */
	var $mConditions = array(
	'always'                      => array(CD_ALWAYS),
	'never'                       => array(CD_ALWAYS, true),
	'ifloggedin'                  => array(CD_IFLOGGEDIN),
	'ifnotloggedin'               => array(CD_IFLOGGEDIN, true),
	'ifhasright'                  =>
	  array(CD_IFHASRIGHT, false, true, '(', ')', true, '"'),
	'ifnothasright'               =>
	  array(CD_IFHASRIGHT, true , true, '(', ')', true, '"'),
	'ifisarticlepage'             => array(CD_IFISARTICLE),
	'ifnotisarticlepage'          => array(CD_IFISARTICLE, true),
	'ifisnotarticlepage'          => array(CD_IFISARTICLE, true), # alias
	'ifistalkpage'                => array(CD_IFISTALK),
	'ifnotistalkpage'             => array(CD_IFISTALK, true),
	'ifisnottalkpage'             => array(CD_IFISTALK, true), # alias
	'ifisspecialpage'             => array(CD_IFISSPECIAL),
	'ifnotisspecialpage'          => array(CD_IFISSPECIAL, true),
	'ifisnotspecialpage'          => array(CD_IFISSPECIAL, true), # alias
	'ifpageexists'                => array(CD_IFPAGEEXISTS),
	'ifnotpageexists'             => array(CD_IFPAGEEXISTS, true),
	'ifarticlepageexists'         => array(CD_IFARTICLEEXISTS),
	'ifnotarticlepageexists'      => array(CD_IFARTICLEEXISTS, true),
	'iftalkpageexists'            => array(CD_IFTALKEXISTS),
	'ifnottalkpageexists'         => array(CD_IFTALKEXISTS, true),
	'ifisforeignuserpage'         => array(CD_IFISFOREIGNUSERPAGE),
	'ifnotisforeignuserpage'      => array(CD_IFISFOREIGNUSERPAGE,true),
	'ifisnotforeignuserpage'      => array(CD_IFISFOREIGNUSERPAGE,true),#ali
	'ifisemailableuserpage'       => array(CD_IFISEMAILABLEUSERPAGE),
	'ifnotisemailableuserpage'    => array(CD_IFISEMAILABLEUSERPAGE,true),
	'ifisnotemailableuserpage'    => array(CD_IFISEMAILABLEUSERPAGE,true),#a
	'ifviewingoldtalkpage'        => array(CD_IFVIEWINGOLDTALKPAGE),
	'ifnotviewingoldtalkpage'     => array(CD_IFVIEWINGOLDTALKPAGE, true),
	'ifviewingoldarticlepage'     => array(CD_IFVIEWINGOLDARTICLEPAGE),
	'ifnotviewingoldarticlepage'  => array(CD_IFVIEWINGOLDARTICLEPAGE,true),
	'ifpageisdeleted'             => array(CD_IFPAGEISDELETED),
	'ifnotpageisdeleted'          => array(CD_IFPAGEISDELETED, true),
	'ifpageisdeleted_predir'      => array(CD_IFPAGEISDELETED_PREDIR),
	'ifnotpageisdeleted_predir'   => array(CD_IFPAGEISDELETED_PREDIR, true),
	'ifarticlepageisdeleted'      => array(CD_IFARTICLEPAGEISDELETED),
	'ifnotarticlepageisdeleted'   => array(CD_IFARTICLEPAGEISDELETED, true),
	'iftalkpageisdeleted'         => array(CD_IFTALKPAGEISDELETED),
	'ifnottalkpageisdeleted'      => array(CD_IFTALKPAGEISDELETED, true),
	'ifarticlepageisprotected'    => array(CD_IFARTICLEPAGEISPROTECTED),
	'ifnotarticlepageisprotected' =>array(CD_IFARTICLEPAGEISPROTECTED,true),
	'iftalkpageisprotected'       => array(CD_IFTALKPAGEISPROTECTED),
	'ifnottalkpageisprotected'    => array(CD_IFTALKPAGEISPROTECTED, true),
	'ifwatchingpage'              => array(CD_IFWATCHINGPAGE),
	'ifnotwatchingpage'           => array(CD_IFWATCHINGPAGE, true),
	'ifredirected'                => array(CD_IFREDIRECTED),
	'ifnotredirected'             => array(CD_IFREDIRECTED, true),
	'ifisarticlepage_predir'      => array(CD_IFISARTICLE_PREDIR),
	'ifnotisarticlepage_predir'   => array(CD_IFISARTICLE_PREDIR, true),
	'ifistalkpage_predir'         => array(CD_IFISTALK_PREDIR),
	'ifnotistalkpage_predir'      => array(CD_IFISTALK_PREDIR,true),
	'ifarticlepageexists_predir'  => array(CD_IFARTICLEEXISTS_PREDIR),
	'ifnotarticlepageexists_predir'=>array(CD_IFARTICLEEXISTS_PREDIR, true),
	'iftalkpageexists_predir'     => array(CD_IFTALKEXISTS_PREDIR),
	'ifnottalkpageexists_predir'  => array(CD_IFTALKEXISTS_PREDIR, true),
	'ifnodeidmatches'             => array(CD_IFNODEIDMATCHES),
	'ifnotnodeidmatches'          => array(CD_IFNODEIDMATCHES, true),
	'iftitlematches'              => array(CD_IFTITLEMATCHES),
	'ifnottitlematches'           => array(CD_IFTITLEMATCHES, true),
	'ifurlextramatches'           => array(CD_IFURLEXTRAMATCHES),
	'ifnoturlextramatches'        => array(CD_IFURLEXTRAMATCHES, true),
	'ifurlcontainsextra'          => array(CD_IFURLCONTAINSEXTRA),
	'ifnoturlcontainsextra'       => array(CD_IFURLCONTAINSEXTRA, true),
	'ifurlquerycontains'          =>
	  array(CD_IFURLQUERYCONTAINS, false, true, '?', array(' ',"\t",'')),
	'ifnoturlquerycontains'       =>
	  array(CD_IFURLQUERYCONTAINS, true , true, '?', array(' ',"\t",'')),
	/** REMINDER:: document additions, deletions and modifications in
	 * help-articles/Help.Treeview_skin.Syntax.wiki */
	);

	var $mDefaultStartDelimiters = '(';
	var $mDefaultEndDelimiters   = ')';

	/**#@+ @access public */
	function __construct($state = null) {
		$this->mState = $state ? $state : new State;
	}

	/** Maps a string condition as might appear in a user-editable config
	  * page to a five element array and returns that array with meaning:
	  * [0] => an integer constant representing the base condition type.
	  * [1] => true if this represents the negation of the base condition.
	  * [2] => true if the condition takes a parameter.
	  * [3] => parameter start delimiter(s) as an array of character(s)
	  * [4] => parameter end delimiter(s) as an array of character(s); an
	  *        empty character indicates that the parameter can end
	  *        implicitly with the end-of-condition(s) delimiter.
	  * [5] => true if the parameter may contain variable replacements.
	  * [6] => the quote character that may optionally delimit the parameter
	  *        and that may be escaped by doubling; if empty then none
	  *        applies; in any case quotes are optional unless the
	  *        parameter contains one of the end delimiters of [4].
	  * For elements 1 and 2, null or unset might be used in place of false.
	  * An empty result will be returned if the condition string is unknown
	  * to this class.
	  * All elements are always returned even when [3] and [4] are
	  * inapplicable: this allows the function return to be assigned to a
	  * list() without E_NOTICE warnings.
	  */
	function getCondition($conditionStr) {
		$ret = $this->mConditions[$conditionStr];
		/* Avoid E_NOTICE warnings in callers. */
		if (!$ret) $ret = array(null, null, null);
		else {
			if (!isset($ret[1])) $ret[1] = false;
			if (!isset($ret[2])) $ret[2] = false;
		}
		if (!isset($ret[3])) $ret[3] = $this->mDefaultStartDelimiters;
		$ret[3] = (array)$ret[3];
		if (!isset($ret[4])) $ret[4] = $this->mDefaultEndDelimiters;
		$ret[4] = (array)$ret[4];
		if (!isset($ret[5])) $ret[5] = true;
		if (!isset($ret[6])) $ret[6] = '';
		return $ret;
	}

	function getAllParamStartDelims() {
		static $ret = null;
		if (!$ret) {
			$ret = (array)$this->mDefaultStartDelimiters;
			foreach ($this->mConditions as $condArr) {
				if (!empty($condArr[3])) {
					$ret = array_merge($ret,
					  (array)$condArr[3]);
				}
			}
		}
		return $ret;
	}

	/** @static */
	static function isUserPage($nt) {
		return ($nt->getNamespace() == NS_USER_TALK ||
			$nt->getNamespace() == NS_USER);
	}

	/** @static */
	static function getFirstSubPageOrContext($nt) {
		$arr = preg_split('/[\/:]/', $nt->getText());
		$arr = array_reverse($arr);
		return array_pop($arr);
	}

	function isForeignUserPage($nt) {
		$user = $this->mState->getUser();
		if ($this->isUserPage($nt)) {
			$name = $this->getFirstSubPageOrContext($nt);
			if ($name && ($name != $user->getName()) &&
			    ($user = User::newFromName($name)) &&
			    $user->getId()
			) {
				return $name;
			} else return false;
		} else	return false;
	}

	/**
	 * Tests each condition, stopping at the first one that tests false.
	 * Returns true if all conditions hold or if there are no conditions,
	 * otherwise returns false.
	 * @param Array $conditions An array of conditions, each member itself
	 *  a three-member array whose members correspond to the parameters of
	 *  testCondition().
	 * @param mixed $defaultExtra The third argument to supply to
	 *  testCondition() for any member of $conditions whose third member
	 *  is null or unset.
	 * @return boolean.
	 */
	function testConditions($conditions, $defaultExtra = null) {
		$conditions = (array)$conditions;
		foreach ($conditions as $condition) {
			list($condCode, $negate, $extra) = $condition;
			if (is_null($extra)) $extra = $defaultExtra;
			if (!$this->testCondition($condCode, $negate, $extra)) {
				return false;
			}
		}
		return true;
	}

	/**
	 * Tests the condition, returning true if it is satisfied, false if not.
	 * The intended semantics of each condition are documented in:
	 * help-articles/Help.Treeview_skin.Syntax.wiki.
	 * @param integer $condition The condition code, matching one of CD_IF*.
	 * @param boolean $negate True if the default sense of the condition's
	 *                        test should be inverted.
	 * @param Mixed $extra Any extra test-specific data; only required if
	 *                     the third member of the condition's entry in
	 *                     mConditions is true.  Currently applies to:
	 *   CD_IFHASRIGHT         - $extra should be a rights string
	 *   CD_IFNODEIDMATCHES    - $extra should be an integral id or an
	 *                           array with such a member keyed by 'id'
	 *   CD_IFTITLEMATCHES     - $extra should be a string title or an array
	 *                           with such a member keyed by 'title'
	 *   CD_IFURLEXTRAMATCHES  - $extra should be an array with keys 'url'
	 *                           and 'urlextra' whose values are strings
	 *                           that correspond respectively to a full url
	 *                           produced by the Title class, and the
	 *                           'urlextra' parameter for NODE directives in
	 *                           the Hierarchy configuration i.e. formatted
	 *                           as the part of a url that follows the ?.
	 *                           The point of the 'url' key is to catch the
	 *                           implicit action= of urls that match entries
	 *                           in $wgActionPaths.
	 *   CD_IFURLCONTAINSEXTRA - as for CD_IFURLEXTRAMATCHES.
	 *   CD_IFURLQUERYCONTAINS - $extra should be an effective url query
	 *                           string whose parameters and values are
	 *                           url-encoded i.e. equivalent in form to the
	 *                           'urlextra' key above.
	 * @return boolean.
	 */
	function testCondition($condition, $negate = false, $extra = null) {
		$fname = 'Conditions::testCondition';
		$user = $this->mState->getUser();
		$tPage = $this->mState->getTitle();
		$tPage_pre = $this->mState->getTitle_Predir();
		$webRequest = $this->mState->getWebRequest();

		$positive = !$negate;
		switch ($condition) {
		case CD_ALWAYS:
			$ret = $positive;
			break;
		case CD_IFLOGGEDIN:
			$ret = ($user->isRegistered() == $positive);
			break;
		case CD_IFHASRIGHT:
			$ret = ($user->isAllowed($extra) == $positive);
			break;
		case CD_IFISARTICLE:
		case CD_IFISARTICLE_PREDIR:
			$t = $condition==CD_IFISARTICLE ? $tPage : $tPage_pre;
			$ret = (($t->getNamespace() != NS_SPECIAL &&
			  !$t->isTalkPage()) == $positive);
			break;
		case CD_IFISTALK:
		case CD_IFISTALK_PREDIR:
			$t = $condition==CD_IFISTALK ? $tPage : $tPage_pre;
			$ret = ($t->isTalkPage() == $positive);
			break;
		case CD_IFISSPECIAL:
			$ret = (($tPage->getNamespace() == NS_SPECIAL) ==
			  $positive);
			break;
		case CD_IFPAGEEXISTS:
			$ret = (($tPage->getNamespace() == NS_SPECIAL ||
			  $tPage->exists()) == $positive);
			break;
		case CD_IFARTICLEEXISTS:
		case CD_IFARTICLEEXISTS_PREDIR:
			$t = $condition==CD_IFARTICLEEXISTS?$tPage:$tPage_pre;
			$tArt = $t->getSubjectPage();
			$ret = (($tArt && $tArt->exists()) == $positive);
			break;
		case CD_IFTALKEXISTS:
		case CD_IFTALKEXISTS_PREDIR:
			$t = $condition==CD_IFTALKEXISTS ? $tPage : $tPage_pre;
			$tTalk = $t->getTalkPage();
			$ret = (($tTalk && $tTalk->exists()) == $positive);
			break;
		case CD_IFISFOREIGNUSERPAGE:
		case CD_IFISEMAILABLEUSERPAGE:
			$ret = ($this->isForeignUserPage($tPage) == $positive);
			break;
		case CD_IFVIEWINGOLDTALKPAGE:
			$ret = (($tPage->isTalkPage() &&
			  $webRequest->getVal('oldid') &&
			  !$webRequest->getVal('diff')) == $positive);
			break;
		case CD_IFVIEWINGOLDARTICLEPAGE:
			$ret = ((($tPage->getNamespace() != NS_SPECIAL &&
			  !$tPage->isTalkPage()) &&
			  $webRequest->getVal('oldid') &&
			  !$webRequest->getVal('diff')) == $positive);
			break;
		case CD_IFPAGEISDELETED:
		case CD_IFPAGEISDELETED_PREDIR:
			$t = $condition==CD_IFPAGEISDELETED?$tPage:$tPage_pre;
			$ret = ((!$t->getArticleId() && $t->isDeleted()) ==
			  $positive);
			break;
		case CD_IFARTICLEPAGEISDELETED:
		case CD_IFTALKPAGEISDELETED:
			$t = $condition == CD_IFARTICLEPAGEISDELETED ?
			  $tPage->getSubjectPage() : $tPage->getTalkPage();
			$ret = (($t && !$t->getArticleId() && $t->isDeleted())==
			  $positive);
			break;
		case CD_IFARTICLEPAGEISPROTECTED:
		case CD_IFTALKPAGEISPROTECTED:
			$t = $condition == CD_IFARTICLEPAGEISPROTECTED ?
			  $tPage->getSubjectPage() : $tPage->getTalkPage();
			$t = $tPage->getSubjectPage();
			$ret = (($t && $t->exists() &&
			  MediaWikiServices::getInstance()->getRestrictionStore()->isProtected($t)) == $positive);
			break;
		case CD_IFWATCHINGPAGE:
			$ret = (MediaWikiServices::getInstance()->getWatchlistManager()->isWatched(RequestContext::getMain()->getUser(), $tPage) == $positive);
			break;
		case CD_IFREDIRECTED:
			$ret = (($this->mState->redirected() &&
			  $webRequest->getVal('redirect')!='no') == $positive);
			break;
		case CD_IFNODEIDMATCHES:
			/* $extra might be a $node aka a $vn */
			$id = is_array($extra) ? $extra['id'] : $extra;
			$ret = ($positive == ($id ==
			  $webRequest->getVal('nodeid')));
			break;
		case CD_IFTITLEMATCHES:
			/* $extra might be a $node aka a $vn */
			$title = is_array($extra) ? $extra['title'] : $extra;
			/* Match against the "alias" if viewing a redirect. */
			$ret = (($tPage_pre->getPrefixedText() == $title) ==
			  $positive);
			break;
		case CD_IFTITLEMATCHES_POSTREDIR:
			/* $extra might be a $node aka a $vn */
			$title = is_array($extra) ? $extra['title'] : $extra;
			/* Match against the "alias" if viewing a redirect. */
			$ret = (($tPage->getPrefixedText() == $title) ==
			  $positive);
			break;
		case CD_IFURLEXTRAMATCHES:
		case CD_IFURLCONTAINSEXTRA:
			/* There might be unhandled corner-cases here where one
			 * of $wgActionPaths has a legal, unlisted alternative
			 * such as a trailing / but in practice these are
			 * likely to be rare and the effect of a false negative
			 * probably doesn't warrant the additional complexity
			 * and configuration that would be required to avoid it.
			 */
			global $wgActionPaths;
			$action = $wgActionPaths ?
			  array_search($extra['url'], $wgActionPaths) : false;
			if (empty($extra['urlextra'])) $extra['urlextra'] = '';
			$testArr = $action ? array('action' => $action) :
			  TvQueryString::queryStrToArr($extra['urlextra']);
			/* fall through */
		case CD_IFURLQUERYCONTAINS:
			$qArr = $this->mState->getQueryArr(true);
			if ($condition == CD_IFURLQUERYCONTAINS) {
				$testArr = TvQueryString::queryStrToArr($extra);
			} /* else set in case above */
			if (($condition == CD_IFURLCONTAINSEXTRA ||
			     $condition == CD_IFURLQUERYCONTAINS) &&
			  !isset($testArr['action']) &&
			   isset($qArr   ['action']) &&
			  !in_array($qArr['action'], TvQueryString::
			    getViewSynonyms())) {
				/* If the 'action' query parameter is not
				 * specified within urlextra, and if the url's
				 * query parameters contain an 'action' with a
				 * value other than 'view', then this should be
				 * considered to 'contain' the unspecified
				 * 'action' of urlextra: avoid inserting 'view'.
				 */
			} else $testArr = TvQueryString::convertViewAction(
			  $testArr, 'view');
			if ($condition == CD_IFURLEXTRAMATCHES) {
				$res = ($testArr == $qArr);
			} else	$res = (array_diff_assoc($testArr, $qArr) ?
			  false : true);
			$ret = ($res == $positive);
			break;
		default:
			wfDebug("$fname: received an unknown condition ".
			  "constant with value: \"$condition\"\n");
			$ret = false;
		}
		return $ret;
	}
	/**#@- */ # end public section
}
?>
