<?php
/**
 * Hierarchy.php
 *
 * The main file in the Hierarchy extension for MediaWiki.  This extension
 * supports the Treeview skin, which relies upon it and which this extension
 * assumes the presence of.  Provided functionality: reads and parses the
 * treeview config file, builds and caches the hierarchy for xhtml output.
 *
 * Licenced under the General Public Licence 2, without warranty.
 *
 * @licence GPL2 http://www.gnu.org/copyleft/gpl.html
 * @author Laird Shaw <lairdshaw77@gmail.com>
 */

if (!defined('MEDIAWIKI')) die();

use MediaWiki\MediaWikiServices;
use MediaWiki\Title\Title;
use MediaWiki\Linker\Linker;

# in lieu of DefaultSettings.php entries
if (!isset($wgTreeviewIconWidth)) $wgTreeviewIconWidth = 13;
if (!isset($wgTreeviewIconHeight)) $wgTreeviewIconHeight = 13;

/** Version the cached object - an idea copied from MW core code */
define('HIERARCHY_OBJ_VERSION', '1');
/** Treeview root id - for the node with type 'ROOT' */
define('TV_ROOT_ID', '_ROOT');
/** Leaf, expand and collapse icons, in that order. */
define('ICONS_PER_NODE', 3);
/** Values for the $whichConditions parameter to doNodeReplacements() and/or
  * the $which parameter to doConditionsReplacements() */
define('TV_CONDREPL_NONE'  , 0);
define('TV_CONDREPL_PARAMS', 1);
define('TV_CONDREPL_NODE'  , 2);
define('TV_CONDREPL_BOTH'  , TV_CONDREPL_PARAMS | TV_CONDREPL_NODE);

global $IP; # not required for MW 1.5.x or 1.6.x
/** For several array/string helpers, such as x_array_key_values() */
require_once('UtilityFunctions.php');
/** For the Conditions constructor */
require_once('Conditions.php');
/** For the TVState constructor */
require_once('State.php');
/** For TvQueryString::getViewSynonyms(),::queryStrToArr(),::arrToQueryStr(). */
require_once('QueryString.php');
/** For wfUrlProtocols() */
require_once("$IP/includes/GlobalFunctions.php");

global $wgTV_RedirectedFrom;
$wgTV_RedirectedFrom = false;

function wfHierarchy_RenderTreeviewTag($text, $params = array()) {
	global $wgHierarchy, $wgArticle, $wgTitle,
	  $wgSpecialHierarchySavedHookData;

	$wgSpecialHierarchySavedHookData['treeviewtagran'] = true;
	if (!isset($wgHierarchy)) $wgHierarchy = new Hierarchy();
	$pruneids = isset($params['pruneids']) ?
	  explode(',', $params['pruneids']) : array();
	if (isset($params['showtitle'])) {
		$t = Title::newFromText($params['showtitle']);
		$showtitle_parsed = $t->getPrefixedText();
	} else if (isset($params['showthis'])) {
		$t = ($wgArticle ? $wgArticle->getTitle() : $wgTitle);
		$showtitle_parsed = $t->getPrefixedText();
	} else	$showtitle_parsed = null;

	return $wgHierarchy->renderTreeviewTag($showtitle_parsed, $pruneids);
}

MediaWikiServices::getInstance()->getParser()->setHook('treeview', 'wfHierarchy_RenderTreeviewTag');

/**
 * Invalidates every treeview-related cache and all articles in the
 * file cache and client-caches; avoids the parser cache.
 */
function wfHierarchy_invalidateAll() {
	$services = MediaWiki\MediaWikiServices::getInstance();
	$wgMemc = ObjectCache::getLocalClusterInstance();
	wfDebug(__METHOD__.": INVALIDATING the cache of all articles ".
	  "+ the ".'treeview hierarchy + the remappings list'."\n");
	wfHierarchy_invalidateAll_File_Client();
	$key = Hierarchy::getHierarchyKey();
	$wgMemc->delete($key);
	wfLogTreeview('Deleted the hierarchy key ('.$key.
	  ') from the cache.'."\n");
	$key = Hierarchy::getRemappingsKey();
	$wgMemc->delete($key);
	wfLogTreeview('Deleted the remappings key ('.$key.
	  ') from the cache.'."\n");
}

/**
 * Invalidates all articles in the file cache and client-caches;
 * avoids the parser cache.
 */
function wfHierarchy_invalidateAll_File_Client() {
	global $wgCacheEpochTouchFile, $wgUseFileCache, $wgCachePages;
	if (!@touch($wgCacheEpochTouchFile)) {
		wfLogTreeview('Touched the cache epoch file ('.
		  $wgCacheEpochTouchFile.').'."\n");
		if ($wgUseFileCache || $wgCachePages) {
			global $wgOut;
			$wgOut->showFatalError(wfMessage(
			  'tv_untouchable', __METHOD__)->escaped());
		}
	}
}

/**
 * The one and only non-configuration global used by Treeview-related code: it's
 * used only to work around the parts of MediaWiki's hook system that aren't
 * designed for Treeview purposes.
 */
global $wgSpecialHierarchySavedHookData;
$wgSpecialHierarchySavedHookData = array('invalidatedall' => false);

/**
 * This class produces and maintains the hierarchy used by the Treeview skin,
 * as well as by the Hierarchy special page defined by the HierarchyPage class
 * in SpecialHierarchy.php.
 *
 * @package MediaWiki
 * @subpackage Extensions
 */
class Hierarchy {
	/**#@+ @access private */
	/** The config-page parse tree, heavily processed post-parsing and
	 * finally used to cache html fragments of the treeview.  This entire
	 * object is cached between requests if a MW cache is configured.
	 */
	var $mHierarchy;
	/** A tree based on colon-separated title contexts for this wiki as
	 * queried from the database; contains at a minimum all of the titles
	 * required for importing per the config page directives but doesn't
	 * take prune directives into account so for a huge wiki this tree's
	 * size (and DB querying) will be a problem.
	 */
	var $mHierarchyDB;
	/** A 2-d array keyed on full visible context; initialised from the
	 * DB table {$wgDBprefix}treeview_used_nodeids if
	 * $wgStoreTreeviewNodeIdsInDB is true. */
	var $mContextNodeIds = array();
	/** A flat array that keys nodeids to full visible contexts; kept
	 * consistent with mContextNodeIds */
	var $mNodeIdContexts = array();
	/** A list of new ids for direct insertion into the database. */
	var $mNewNodeIds = array();
	/** A list keyed by ids of 'nodeid' parameters, values are an array of
	  * 0 => line #, 1 => if-else group #. */
	var $mNodeIdsParsed = array();
	/** Subcontext underscores-in-title-remapping list, indexed by title as
	 * returned by Title::getPrefixedText() */
	var $mRemappings;
	/**
	 * The parsing rules used in parseConfigPage() and elsewhere; some
	 * additions would require changes there or elsewhere - in particular:
	 * + adding 'title' to the 'replaceables' list of 'antiprune', 'prune'
	 *   or 'import';
	 * + adding 'start' or 'last' to the 'replaceables' list of 'block';
	 * + setting 'conditionable' true for 'import' or 'block' - and without
	 *   changes elsewhere this also has unintuitive behaviour for 'prune'
	 *   and 'antiprune' directives.
	 */
	var $mParseRules = array(
		'node'         => array(
			'conditionable' => true,
			'replaceables'  => array('url', 'title', 'dispname',
			  'urlextra', 'tooltip'),
			'parameters'    => array('nomatch', 'nodeid',
			  'knownlink', 'appendnodeid', 'selected', 'urlextra',
			  'tooltip', 'expanded', 'icons')),
		'antiprune'    => array(
			'conditionable' => false,
			'replaceables'  => array('tooltip'),
			'parameters'    => array('nomatch', 'nodeid',
			  'pruneleafs', 'knownlink', 'appendnodeid', 'selected',
			  'merge', 'tooltip', 'expanded', 'icons')),
		'import'       => array(
			'conditionable' => false,
			'replaceables'  => array(),
			'parameters'    => array('nomatch', 'pruneleafs',
			  'knownlink', 'appendnodeid', 'merge', 'selected',
			  'expanded', 'icons'),
			'nopropagate'   => array('pruneleafs')),
			/* 'propagated' => all parameters except those in
			 *   nopropagate; this is set in the constructor. */
		/*'category' => set in the constructor to match 'import' + 'showpages' parameter. */
		'prune'        => array(
			'conditionable' => false,
			'replaceables'  => array('tooltip'),
			'parameters'    => array('nomatch', 'nodeid',
			  'pruneleafs', 'knownlink', 'appendnodeid', 'selected',
			  'merge', 'tooltip', 'expanded', 'icons')),
		'block'        => array(
			'conditionable' => false,
			'replaceables'  => array(),
			'parameters'    => array('exclude')),
		'endimport'    => array('replaceables' => array()),
		'null'         => array('replaceables' => array()),
		'importednode' => array('replaceables' => array())
	);
	/** The 'conditionable' rule is only applied to parameters that are
	  * without a 'splitter'.
	  */
	var $mParameterOptions = array(
		'nomatch'      => array(),
		'pruneleafs'   => array(),
		'knownlink'    => array(),
		'appendnodeid' => array(),
		'expanded'     => array(),
		'showpages'    => array(),
		'selected'     => array(
			'conditionable' => true
		),
		'nodeid'       => array(
			'splitter'      => '=',
			'key'           => 'id'
		),
		# never used - nodeid was used for imports
//		'baseid' => array('splitter' => '=', 'key' => 'baseid'),
		'urlextra'     => array(
			'splitter'      => '?'
		),
		'merge'        => array(
			'splitter'      => '=',
			'values'        => array('sort', 'prepend', 'append')
		),
		'tooltip'      => array(
			'splitter'      => '=',
			'quotechar'     => '"',
			'msgkeyable'    => true,
		),
		'exclude'      => array(
			'splitter'      => '=',
			'values'        => array('first', 'last', 'both')
		),
		/* Additional special-case handling is in parseParameters(). */
		'icons'        => array(
			'splitter'      => '='
		),
	);

	/** @var TVstate */
	var $mState;
	/** @var Conditions */
	var $mConditions;

	/** Set true after updateTreeviewState_r() has been run on the
	  * loaded/generated hierarchy */
	var $mIsCurrent;

	/** The value returned by the last call to parseConfigPage() during this
	  * object's lifetime.  A non-null value implies that the value of the
	  * following members is meaningful and was derived from the parse:
	  * mHierarchy, mFirstLineNum, mParseErrors, mParseErrors_Html,
	  * mParseSql, mRemappings.
	  * Set it to null if any of those members change later - that way if
	  * parseConfigPage() is called again then it will re-parse rather than
	  * rely on the validity and consistency of those stored members.
	  * @var boolean
	  */
	var $mParseResult = null;

	/** The number of the line with respect to which other line numbers are
	  * counted when reporting errors/warnings.  Counts start at 1, not 0.*/
	var $mFirstLineNum = 1;
	/** Plain-text newline-delimited string; this is set iff errors occur.*/
	var $mParseErrors = '';
	/** As for mParseErrors except an html <br />-delimited string. */
	var $mParseErrors_Html = '';
	/** The minimal(ish) query conditions determined by parseConfigPage().*/
	var $mParseSql = null;

	/** Used solely by getConfigPageText() as a cache.  Can at any point be
	  * set to null to reclaim memory, with the risk of a re-fetch. */
	var $mConfigText = null;
	/** Used solely by readConfigPage() as a cache.  Can at any point be
	  * set to null to reclaim memory, with the risk of re-processing. */
	var $mConfPageData = null;

	var $mDefaultIcons;
	var $mExpandeds;

	/**#@- */

	/**#@+ @access public */
	/** Returns the hierarchy structure adjusted to the current request
	  * i.e. with replacements made and conditions applied.
	  */
	function getCurrentHierarchy() {
		if (!$this->mIsCurrent) {
			$vn =& $this->getHierarchy();
			$this->updateTreeviewState_r($vn);
			$this->mIsCurrent = true;
		}
		return $this->mHierarchy;
	}

	/**
	 * Retrieves all messages referenced by the treeview config page
	 * i.e. all messages for which a change should be reflected in the
	 * treeview.  Call statically only: a temporary Hierarchy object will
	 * be created and then dropped.
	 * @return Array of message keys, each as would be passed to wfMessage().
	 * @static
	 */
	static function getMessageDeps() {
		$hObj = new Hierarchy;
		$hierarchy = $hObj->getHierarchy();
		return array_keys($hierarchy['message_deps']);
	}

	/**
	 * Returns true if the image in the "Image" (or equivalent) namespace
	 * with title $imageName is used as an icon in the treeview.  Call
	 * statically only: a temporary Hierarchy object will be created and
	 * then dropped.
	 * @return Boolean
	 * @static
	 */
	static function usesImage($imageName) {
		$hObj = new Hierarchy;
		return $hObj->recurseHierarchy('isImageInNode',
		  $hObj->getHierarchy(), false, $imageName) ? true : false;
	}

	/** @todo document */
	function getRemappings() {
		if (!$this->mRemappings && !$this->loadRemappings()) {
			$this->readConfigPage($instructions, $remappings, $x);
			/* parseConfigPage() might have called this already. */
			$this->validateRemappings($remappings, true);
			/* getHierarchy() might have called this already. */
			$this->saveRemappings(true);
		}
		return $this->mRemappings;
	}

	/** @todo document */
	function __construct($state = null) {
		$this->mState = $state ? $state : new TVstate;
		$this->mConditions = new Conditions($this->mState);
		$this->setDefaultIcons();
		$r =& $this->mParseRules['import'];
		$r['propagated']=array_diff($r['parameters'],$r['nopropagate']);
		$this->mParseRules['category'] = $this->mParseRules['import'];
		$this->mParseRules['category']['parameters'][] = 'showpages';
	}

	/**
	 * Sets the default leaf, plus and minus icons for the treeview, as
	 * passed in an array in that order.
	 * @param Array $icons Each entry is a (local) url to an image.
	 */
	function setDefaultIcons($icons = array()) {
		static /* const */ $fallbackSkinName = 'treeview';
		if (!$icons) {
			$skin = RequestContext::getMain()->getSkin();
			if (method_exists($skin, 'getDefaultTVicons')) {
				$icons = $skin->getDefaultTVicons();
			} else	wfDebug("Hierarchy::setDefaultIcons: ".
				  "could not find a skin with a ".
				  "'getDefaultTVicons' method\n");
		}
		$this->mDefaultIcons = $icons;
	}

	/** @todo document */
	function getExpandedNodesStr() {
		return implode(',', $this->getExpandedNodes());
	}

	/** @todo document */
	function getExpandedNodes() {
		return array_unique($this->mExpandeds);
	}

	/** @todo document */
	function getParseErrors_Html() {
		return $this->mParseErrors_Html;
	}

	function renderTreeviewTag($titletoshow, $pruneids) {
		$hierarchy = $this->getCurrentHierarchy();
		$ret = '';
		$id = null;
		for ($i = 0; $i < count($hierarchy['nodes']); $i++) {
			if (!empty($hierarchy['nodes'][$i]['id'])) {
				$id = $hierarchy['nodes'][$i]['id'];
			}
			$altdispname = null;
			if ($titletoshow !== null) {
				$t = Title::newFromText($titletoshow);
				if ($t !== null) {
					$a = explode(':', $t->getText());
					$altdispname = $a[count($a) - 1];
				}
			}
			$ret .= $this->renderTreeviewTag_r(
			  $hierarchy['nodes'][$i], $pruneids, $titletoshow,
			  $altdispname, $id, $titletoshow === null);
		}
		return $ret;
	}

	function renderTreeviewTag_r($vn, $pruneids, $titletoshow, $altdispname,
	  $id, $show = false) {
		$ret = '';

		static $aids = array();

		if (!empty($id) && in_array($id, $pruneids)) return '';
		if ($titletoshow == $vn['title'] &&
		    ($titletoshow == $vn['dispname'] ||
		     $altdispname == $vn['dispname'])) {
			$show = true;
		}
		if ($show) {
			$ret .= '<ul>'."\n";
			$ret .= '<li>'."\n";
			if (empty($vn['link'])) $this->realiseNodeLink($vn, '', $aids);
			$ret .= $vn['link']."\n";
		}
		if ($vn['nodes']) {
			$id2 = null;
			for ($i = 0; $i < count($vn['nodes']); $i++) {
				if (!empty($hierarchy['nodes'][$i]['id'])) {
					$id2 = $hierarchy['nodes'][$i]['id'];
				}
				$ret .= $this->renderTreeviewTag_r(
				  $vn['nodes'][$i], $pruneids, $titletoshow,
				  $altdispname, $id2, $show);
			}
		}
		if ($show) {
			$ret .= '</li>'."\n";
			$ret .= '</ul>'."\n";
		}
		return $ret;
	}

	/**
	 * $nodeIds can be empty (rebuild entire treeview), a single ID or
	 * an array of IDs.  In the case of an arrayed ID input, the return
	 * will also be an array, indexed by ID - otherwise it will be a single
	 * string containing the treeview html.
	 */
	function buildHierarchyHtml($nodeIds = null, $skipStateRebuild = false,
	  $indent = "\t", $ul = false) {
		$this->mExpandeds = array();
		$hierarchy = $this->getCurrentHierarchy();
		$multi = is_array($nodeIds);
		if (!$multi) $nodeIds = array($nodeIds);
		$htmlArr = array();
		foreach ($nodeIds as $nodeId) {
			$found = true;
			if ($nodeId !== null) {
				$vn = $this->findNode($nodeId);
				if (!$vn) {
					$found = false;
					wfDebug(__METHOD__.": NOT FOUND: node with ".
					  "id '$nodeId'\n");
				}
			} else	$vn = $hierarchy;
			$htmlArr[$nodeId] = $found ? $this->
			  buildHierarchyHtml_r($vn, false, $skipStateRebuild,
			    $indent, $ul) : false;
		}
		return $multi ? $htmlArr : array_pop($htmlArr);
	}

	/**
	 * Returns (a copy of) the subtree of $this->mHierarchy starting at the
	 * node with id $nodeId or null/unset if no such node was found.
	 */
	function findNode($nodeId) {
		return $this->recurseHierarchy('nodeIfIdMatch', null, false,
		  $nodeId);
	}

	/** Can be called after buildHierarchyHtml() to get the modified state.
	  */
	function getState() {
		return $this->mState;
	}

	/** Tests whether the input title matches that of the Treeview/Hierarchy
	  * configuration page. Returns (type-significant):
	  * true =>  the title matches that of the configuration page.
	  * 0    =>  no configuration page is set, and database messages
	  *          are disabled, however the title matches the article
	  *          corresponding to the default configuration page message.
	  * false => no match whatsoever and/or $wgHierarchyPage is set to an
	  *          invalid name.
	  * @param Title
	  * @return Mixed
	  * @static
	  */
	static function isConfigPage($nt) {
		list($tConfigPage,$fullMatch)= Hierarchy::getConfigPageTitle_();
		if ($tConfigPage) {
			/* Don't match by id because non-existent articles all
			 * have id of 0. */
			$title1 = $nt->getPrefixedText();
			$title2 = $tConfigPage->getPrefixedText();
			$match = ($title1 == $title2);
		} else	$match = false;
		return $match ? ($fullMatch ? true : 0) : false;
	}

	/** Returns the title object of the Treeview/Hierarchy configuration
	  * page, or null if there is no such page or if the page is in one of
	  * the NS_SPECIAL or NS_IMAGE namespaces or was for some unspecified
	  * reason found to be objectionable by the Title class.  If no
	  * configuration page is set, and if database messages are enabled,
	  * and if $maximal is true, then the title of the article corresponding
	  * to the default configuration page message will be returned.  If the
	  * configuration page is set but invalid then false will be returned
	  * instead of null.
	  * @param boolean $maximal
	  * @return Mixed. Title object, null or false.
	  * @note  Keep the 'tv_invconfpg' message synchronised with this
	  * description.
	  * @static
	  */
	static function getConfigPageTitle($maximal = false) {
		list($tConfigPage,$fullMatch)=Hierarchy::getConfigPageTitle_();
		return ($maximal || $fullMatch) ? $tConfigPage : null;
	}

	/** @static */
	static function getConfigPageTitle_() {
		global $wgHierarchyPage, $wgUseDatabaseMessages;
		if (isset($wgHierarchyPage)) {
			$fullMatch = true;
			$tConfigPage = Title::newFromText($wgHierarchyPage);
			if ($tConfigPage->getNameSpace() == NS_SPECIAL ||
			    $tConfigPage->getNameSpace() == NS_IMAGE) {
				$tConfigPage = false;
			}
		} else {
			$fullMatch = $wgUseDatabaseMessages;
			$tConfigPage = Title::makeTitle(NS_MEDIAWIKI,
			  Hierarchy::getDefConfigKey());
		}
		return array($tConfigPage, $fullMatch);
	}

	/** @static */
	static function getDefConfigKey() {
		global $wgCapitalLinks;

		$ret = 'treeview';
		// Later versions of MediaWiki auto-capitalise
		// messages in the MediaWiki namespace.
		if (true/*$wgCapitalLinks*/) $ret = ucfirst($ret);
		return $ret;
	}

	/** @return Mixed. The timestamp of the hierarchy structure, or false if
	  * no hierarchy structure exists or if no timestamp exists. */
	function getHierarchyTs() {
		return isset($this->mHierarchy['generated_timestamp']) ?
		  $this->mHierarchy['generated_timestamp'] : false;
	}

	/**
	 * Loads the hierarchy structure from cache.  Stores any valid
	 * retrieved structure in $this->mHierarchy.  The purpose of this
	 * method's public availability for calls from outside the class is to
	 * check for a cache hit.
	 * @return Mixed: boolean true for a valid cache hit, boolean false for
	 * a cache miss, integer 0 if invalid cache hit due to expired cache
	 * time or object version.
	 */
	function loadHierarchy() {
		global $wgCacheEpoch;

		$services = MediaWiki\MediaWikiServices::getInstance();
		$wgMemc = ObjectCache::getLocalClusterInstance();
		// echo get_class($wgMemc);
		$key = Hierarchy::getHierarchyKey();
		$hierarchy = $wgMemc->get($key);
		$genTime = isset($hierarchy['generated_timestamp']) ? $hierarchy['generated_timestamp'] : 0;
		$cacheHit = empty($hierarchy) ? false : true;
		if (!$cacheHit) {
			$msg = __METHOD__.": hierarchy NOT FOUND in cache using ".
			  "key '$key'\n";
			wfLogTreeview($msg);
			wfDebug($msg);
		} else if ($hierarchy['obj_version'] == HIERARCHY_OBJ_VERSION) {
			/* Save admins the hassle of setting $wgCacheEpoch and
			 * then having to separately trigger a Treeview
			 * invalidation. */
			if ($genTime < $wgCacheEpoch) {
				$cacheHit = 0;
				$msg = __METHOD__.": retrieved cached hierarchy ".
				  "structure from cache using key ".
				  "'$key', but gentime < epoch ".
				  "($genTime < $wgCacheEpoch), so it must be ".
				  "rebuilt.\n";
				wfLogTreeview($msg);
				wfDebug($msg);
			}
		} else {
			$cacheHit = 0;
			$msg = __METHOD__.": IGNORING cached hierarchy structure ".
			  "loaded from cache using key '$key' and versioned ".
			  "as '{$hierarchy['obj_version']}'; this code uses ".
			  "version '".HIERARCHY_OBJ_VERSION."': invoking ".
			  "indiscriminate invalidation for consistency; ".
			  "cache-key used was '$key'\n";
			wfLogTreeview($msg);
			wfDebug($msg);
			wfHierarchy_invalidateAll();
		}
		if ($cacheHit) {
			wfDebug(__METHOD__.": LOADED hierarchy from cache using ".
			  "key '$key'\n");
			$this->mHierarchy = $hierarchy;
			$this->mParseResult = null;
		}
		return $cacheHit;
	}
	/**#@- */ # end public block

	/**#@+ @access private */
	/** @return Mixed (by reference) The hierarchy structure as rebuilt or
	  * retrieved from cache, and unprocessed by updateTreeviewState_r().
	  */
	function &getHierarchy() {
		if (!$this->mHierarchy && !$this->loadHierarchy()) {
			$this->buildHierarchyAndRemappings();
			$this->saveHierarchy();
			/* getRemappings() might have called this already. */
			$this->saveRemappings(true);
		}
		return $this->mHierarchy;
	}

	/** Caches the current state of the hierarchy */
	function saveHierarchy($hierarchy = null) {
		global $wgMainCacheType;
		$services = MediaWiki\MediaWikiServices::getInstance();
		$wgMemc = ObjectCache::getLocalClusterInstance();

		if (empty($hierarchy)) $hierarchy = $this->mHierarchy;
		$key = Hierarchy::getHierarchyKey();
		$wgMemc->set($key, $hierarchy);
		$msg = __METHOD__.": SAVED hierarchy to (a potentially dummy) ".
		  "cache with key '$key' and gentime '{$this->mHierarchy['generated_timestamp']}'\n";
		wfLogTreeview($msg);
		wfDebug($msg);
	}

	/** Saves $this->mRemappings to the MediaWiki-configured main cache.
	  * $checkSetPrior should not be set true for calls other than from the
	  * member functions getHierarchy() and getRemappings().
	  * @param boolean false => save unconditionally and without recording;
	  *           true  => save only if no prior record exists; record save.
	  */
	function saveRemappings($checkSetPrior = false) {
		$services = MediaWiki\MediaWikiServices::getInstance();
		$wgMemc = ObjectCache::getLocalClusterInstance();
		static $priorSave = false;
		if ($checkSetPrior && $priorSave) return;
		$key = Hierarchy::getRemappingsKey();
		$wgMemc->set($key, $this->mRemappings);
		if ($checkSetPrior) $priorSave = true;
		$msg = __METHOD__.": SAVED remappings to (a potentially dummy) ".
		  "cache with key '$key'\n";
		wfLogTreeview($msg);
		wfDebug($msg);
	}

	function loadRemappings() {
		$services = MediaWiki\MediaWikiServices::getInstance();
		$wgMemc = ObjectCache::getLocalClusterInstance();

		$ret = false;
		$key = Hierarchy::getRemappingsKey();

		$remappings = $wgMemc->get($key);
		$cacheHit = $remappings ? true : false;
		if (!$cacheHit) {
			wfDebug(__METHOD__.": title remappings NOT FOUND in cache ".
			  "using key '$key'\n");
		} else {
			$this->mRemappings = $remappings;
			$this->mParseResult = null;
			wfDebug(__METHOD__.": LOADED title remappings from cache ".
			  "using key '$key'\n");
		}
		return $cacheHit;
	}

	/**
	 * Parses the treeview config page and performs post-parse processing
	 * to build a cacheable tree object + title underscore-remappings list.
	 * Makes necessary replacements and builds titles, urls, icons, links,
	 * based on a default state of main page, anonymous user, empty
	 * querystring and nothing selected.
	 */
	function buildHierarchyAndRemappings() {
		global $wgStoreTreeviewNodeIdsInDB;
		# acts as a critical section lock
		if ($wgStoreTreeviewNodeIdsInDB) $this->queryUsedNodeIDs();
		$defState = new TVstate(array(
		  'tEffective'          => -1,   # Main Page
		  'tRedirectInfo'       => null, # unknown: follow on-demand
		  'effectiveUser'       => -1,   # anonymous user
		  'effectiveWebRequest' => -1,   # empty url and query string
		  'TVexpandState'       => -1,   # empty state
		  'autoExpandSelected'  => -1,   # use default, currently true
		  'effectiveQueryStr'   => -1,   # empty string
		  'effectiveAction'     => -1    # use default, currently 'view'
		));
		$currentState = $this->mState;
		$this->mState = $defState;
		$this->parseConfigPage();
		$this->queryDB();
		$this->processParseTree_r($this->mHierarchy);
		$this->mHierarchyDB = null; # for GC clean-up
		$this->recurseHierarchy('initialiseNodeState');
		$this->buildHierarchyHtml_r($this->mHierarchy, true);
		$this->recurseHierarchy('trimNodeForCache');
		$this->mState = $currentState;
		# acts as a critical section unlock
		if ($wgStoreTreeviewNodeIdsInDB) $this->saveNewNodeIDs();
	}

	static function getHierarchyKey() {
		global $wgDBname, $wgDBprefix;
		return "$wgDBname:$wgDBprefix:tv";
	}

	static function getRemappingsKey() {
		global $wgDBname, $wgDBprefix;
		return "$wgDBname:$wgDBprefix:remap";
	}

	/**
	 * Returns the unparsed text of the Hierarchy/Treeview configuration
	 * page, using mConfigText as a cache for calls subsequent to the first.
	 * The "page" is actually a message if both $wgHierarchyPage is unset
	 * and $wgUseDatabaseMessages is false.
	 */
	function getConfigPageText() {
		global $wgHierarchyPage, $wgUseDatabaseMessages;

		$text = null;
		if ($this->mConfigText !== null) {
			$text = $this->mConfigText;
		} else {
			$t = $this->getConfigPageTitle(false/*$maximal*/);
			if ($t === false) {
				$msg = __METHOD__.": \$wgHierarchyPage ".
				  "(\"$wgHierarchyPage\") IS INVALID.  ".
				  "Returning empty string.\n";
				wfLogTreeview($msg);
				wfDebug($msg);
			} else if ($t) {
				$msg = __METHOD__.": sourcing \"".
				  $t->getPrefixedText()."\"\n";
				wfLogTreeview($msg);
				wfDebug($msg);
				if (($wikiPage = MediaWikiServices::getInstance()->getWikiPageFactory()->newFromTitle($t))) {
					$content = $wikiPage->getContent();
					if ($content) {
						$text = str_replace("\r", '',
						  $content->getNativeData());
					}
				}
			}
			if ($text === null) {
				$msg = __METHOD__.": some step in ' +
				  'the series of calls MediaWikiServices::getInstance()->getWikiPageFactory()->newFromTitle(' +
				  '\$this->getConfigPageTitle(false))->' +
				  'getContent()->getNativeData() failed.\n";
				wfLogTreeview($msg);
				wfDebug($msg);
				if ($wgUseDatabaseMessages &&
				  !$wgHierarchyPage) {
					/* On MW >= 1.9.1, msgs are not
					 * stored in the database if not
					 * yet modified. */
					$t = null;
					$msg = __METHOD__.": trying ".
					  "the config message ".
					  "instead\n";
					wfLogTreeview($msg);
					wfDebug($msg);
				}
				$configKey = Hierarchy::getDefConfigKey();
				$msg = __METHOD__.": sourcing the ".
				  "'$configKey' message.\n";
				wfLogTreeview($msg);
				wfDebug($msg);
				$text = wfMessage($configKey)
				          ->inContentLanguage()->plain();
			}
		}
		$this->mConfigText = $text;
		wfLogTreeview(__METHOD__.": retrieved the following config text:\n$text\n");
		return $text;
	}

	/**
	 * Reads the contents of the hierarchy instructions page into the
	 * $instructions and $remappings arrays, performing a small amount of
	 * parsing in the process such that each $instructions member is a
	 * string-key array.  Sets $firstLineNum to the 1-based number of the
	 * first line in the == Hierarchy == section that starts with an
	 * asterisk/hash.
	 * @return boolean True iff a previous call's stored results were used.
	 */
	function readConfigPage(&$instructions, &$remappings, &$firstLineNum) {
		static /* would be const in C */ $HIERARCHY = 1, $REMAP = 2;
		if ($this->mConfPageData !== null) {
			list($instructions, $remappings) = $this->mConfPageData;
			return true;
		}
		$text = $this->getConfigPageText();
		$instructions = $remappings = array();
		$state = $lineNum = $firstLineNum = 0;
		$wantFirstLine = false;
		foreach (explode("\n", $text) as $line) {
			$lineNum++;
			if ($state == $REMAP) {
				$parts = explode('|', $line);
				if (count($parts) < 2) continue;
				# remove initial asterisk
				$remappings[substr($parts[0], 1)] = $parts[1];
			} else if ($state == $HIERARCHY) {
				$d = strspn($line, '*#');
				if ($d) {
					$instructions[] = array(
						'linenum'=> $lineNum,
						'depth'=> $d,
						'stars'=> substr($line, 0, $d),
						'line' => trim(substr($line,$d))
					);
					if ($wantFirstLine) {
						$firstLineNum = $lineNum;
						$wantFirstLine = false;
					}
				}
			}
			if ($state < $HIERARCHY && preg_match(
			  '/^=(=+)(\s*)Hierarchy(\s*)==/',$line)) {
				$state = $HIERARCHY;
				$wantFirstLine = true;
			} else if ($state < $REMAP && preg_match(
			  '/^=(=+)(\s*)Remap(\s*)==/', $line)) {
				$state = $REMAP;
			}
		}
		$this->mConfPageData =
		  array($instructions, $remappings, $firstLineNum);
		/* Can also return true immediately. */
		return false;
	}

	/** Validates the input $remappings array and stores the valid-entry
	  * subset as $this->mRemappings.   $checkSetPrior should not be set
	  * true for calls other than from the member functions
	  * parseConfigPage() or getRemappings().
	  * @param boolean false=>process+store unconditionally, without record
	  *  true  => process+store only if no prior record exists or if
	  *           $this->mRemappings is empty; record as processed.
	  */
	function validateRemappings($remappings, $checkSetPrior = false) {
		static $priorStore = false;
		if ($checkSetPrior && $priorStore && $this->mRemappings) return;
		/* Avoid a false cache miss on retrieval. */
		$this->mRemappings = array(':' => ':');
		foreach ($remappings as $title => $mapping) {
			# remap may not be used to change a displayed title
			# name so that it no longer represents the same title
			# from MW's POV
			$contexts = explode(':', $title);
			$mapTitle = implode(':', array_slice($contexts, 0, -1));
			if ($mapTitle) $mapTitle .= ':';
			$mapTitle .= $mapping;
			$t = Title::newFromText($title);
			$tMap = Title::newFromText($mapTitle);
			if (!$t || !$tMap) continue;
			$validated = $t->getPrefixedText();
			$validatedMapped = $tMap->getPrefixedText();
			if ($validated == $validatedMapped) {
				$this->mRemappings[$validated] = $mapping;
			}
		}
		if ($checkSetPrior) $priorStore = true;
		/* Can also return immediately. */
	}

	/** Precondition: the start delimiter - i.e. '(' - has been chomped
	  * from $lineRem.
	  * @param string $lineRem
	  * @param string $endDelim
	  */
	function parseConditions($lineRem, $endDelim = ')') {
		$conditions = array();
		$last = $err = false;
		while (!$last) {
			$paramDelims_s = $this->mConditions->
			  getAllParamStartDelims();
			$testedDelims = array_merge($paramDelims_s,
			  (array)$endDelim, array(' ', "\t")/* parameters are
			  * w/s separated*/);
			list($pos, $c) = xstrpos($lineRem, $testedDelims);
			if ($pos === false) {
				$err = wfMessage('tv_pe_nocondend',
				  $this->delimMsg($endDelim))->plain();
				break;
			} else if ($c == $endDelim) {
				$ret = ltrim(substr($lineRem, $pos+strlen($c)));
				$last = true;
			}
			$condStr = trim(substr($lineRem, 0, $pos));
			$lineRem = ltrim(substr($lineRem, $pos));
			if (!$condStr && $last) continue;
			list($cond, $negate, $takesparam, $delims_s, $delims_e,
			 $replaceable, $qChar) =
			  $this->mConditions->getCondition($condStr);
			if (!$cond) {
				$err = wfMessage('tv_pe_unknowncond', $condStr)
				  ->plain();
				break;
			}
			if ($takesparam) {
				/*NB $delims_e,$delim_e,$endDelim are distinct*/
				if (in_array('', $delims_e)) {
					$flag = !in_array($endDelim, $delims_e);
					if ($flag) {
						$delims_e = array_merge(
						  array_diff($delims_e,
						             array('')),
						  (array)$endDelim);
					}
				} else	$flag = false;
				$delim_s = whichstarts($lineRem, $delims_s);
				if ($delim_s === null) {
					$err = wfMessage('tv_pe_nocondparamst',
					  $condStr, implode('', $delims_s))
					  ->plain();
					break;
				}
				$offset = $len_s = strlen($delim_s);
				if ($qChar) {
					$tmp = ltrim(substr($lineRem, $len_s));
					if (substr($tmp, 0, strlen($qChar)) !=
					  $qChar) $qParam = false;
					else {
						list($qParam, $lineRem) =
						  $this->parseEscapeable($qChar,
						    $tmp);
						if ($qParam === false) {
							$err = wfMessage('tv_pe_'.
							  'unclosedqt_cond',
							  $qChar, $condStr)
							  ->plain();
							break;
						} else	$offset = 0;
					}
				} else	$qParam = false;
				list($pos, $delim_e) =
				  xstrpos($lineRem, $delims_e, $offset);
				if ($pos === false) {
					$err = wfMessage('tv_pe_nocondparamend',
					  $condStr, $this->delimMsg($delims_e))
					  ->plain();
					break;
				}
				$len_e = strlen($delim_e);
				$param = substr($lineRem,$offset,$pos-$offset);
				$lineRem = ltrim(substr($lineRem, $pos +
				  ($flag ? 0 : $len_e)));
				if ($qParam !== false) {
					if (trim($param) != '') {
						/* This need only be a warning:
						 * parsing can continue; right
						 * now this function isn't set
						 * up to handle that though. */
						$err = wfMessage('tv_pe_'.
						  'unqtpreenddelim', $condStr,
						   $qParam, $param)
						   ->plain();
						break;
					}
					$param = $qParam;
				}
			} else	$param = null;
			$condition = array($cond, $negate, $param);
			/*doConditionReplacements() expects this index to be 3*/
			if ($replaceable) $condition[] = true;
			$conditions[] = $condition;
		}
		return array($ret, $err, $conditions);
	}

	/** Converts an array of delimiters into a message for diagnostics. */
	function delimMsg($delims) {
		$msg = '';
		foreach ($delims as $delim) {
			if ($msg) $msg .= ' ';
			if      ($delim == ' ' ) $msg .= wfMessage('tv_key_space')
			                                  ->plain();
			else if ($delim == "\t") $msg .= wfMessage('tv_key_tab'  )
			                                  ->plain();
			else                     $msg .= $delim;
		}
		return $msg;
	}

	/** @return string. The text of the configuration page, given that
	  * page's key, using the appropriate MediaWiki message function.
	  */
	function configMsgFunc($key) {
		global $wgMainCacheType;
		/* Ensure that individual user preferences can't affect the
		 * language used in the cached object when caching is enabled */
		if ($wgMainCacheType != CACHE_NONE) {
			return wfMessage($key)->inContentLanguage()->plain();
		} else return wfMessage($key)->plain();
	}

	/**
	 * Parses the string in $lineRem to determine which of the starting
	 * specifiers of a node it is.  Returns a list of three elements with
	 * these meanings:
	 * $lineRem - the remaining unparsed component of the function argument,
	 *            with leading whitespace trimmed.
	 * $err - string error message or boolean false if no errors occur;
	 *        $specArr is undefined when $err is not false.
	 * $specArr - an array with these keys:
	 *   'type'     => string; the specifier that $lineRem was found to be.
	 *                 Possible values: 'url','title','subcontext','range'.
	 *   ['type']   => string; the url, title or s/ctxt (unused by 'range')
	 *                 ['type'] refers to the value set in the 'type' key.
	 *   'dispname' => string; the node display name ('url' or 'title' only)
	 *   'first'    => string; the start subcontext of a 'range'
	 *   'last'      => string; the end subcontext of a 'range'.
	 * @return Array($lineRem, $err, $specArr)
	 * @param string $lineRem without leading whitespace - ltrim() first
	 *
	 * @todo review the way that MW core decodes title / url characters and
	 * see how much of that can/should be reused/replicated.
	 * @todo reduce indentation depth, possibly by splitting into reusable
	 * component functions.
	 */
	function parseSpecifier($lineRem) {
		$err = $msgKey = $res = false;
		$specArr = [];
		if ($lineRem[0] != '[' && $lineRem[0] != ',') {
			list($msgKey, $res) = xsplit2(' ', $lineRem);
			$lineRem = trim($this->configMsgFunc($msgKey));
		}
		# If the start of a range is omitted, it's implicitly empty i.e.
		# ,[[[End]]] is treated as [[[]]],[[[End]]]
		# (a provisional and intentionally undocumented feature)
		$comma = ($lineRem[0] == ',');
		$max = 3;
		if (!$comma) $n = strspn($lineRem, '[');
		if ($n == 0 && !$comma) {
			$c = $lineRem[0] ? $lineRem[0]: wfMessage(
			  'tv_non-existent')->plain();
			$err = wfMessage('tv_pe_bad1stspecchar', $c)->plain();
		} else if ($n > $max) {
			$err = wfMessage('tv_pe_toomanystbr',$n,$max)->plain();
		} else {
			if (!$comma) $pos = strpos($lineRem, ']', $n);
			if ($pos === false) {
				$err = wfMessage('tv_pe_unclosedbr')->plain();
			} else if (($n2 = strspn($lineRem, ']', $pos)) != $n &&
			 !$comma) {
				$err = wfMessage('tv_pe_nonmatchbrcnts', $n,
				        $n2)->plain();
			} else {
				if (!$comma) {
					$tmp = substr($lineRem,$n,$pos-$n);
					$lineRem = ltrim(substr($lineRem,
					  $pos + $n));
				}
				if ($n <= 2 && !$comma) {
					if ($n == 1) {
						$sep = ' ';
						$type = 'url';
					} else /* $n == 2 */ {
						$sep = '|';
						$type = 'title';
					}
					$specArr['type'] = $type;
					list($link,$disp) = xsplit2($sep, $tmp);
					$link = trim($link);
					$specArr['dispname'] = trim($disp);
					if ($n == 1 &&
					  !$this->validateExtUrl($link)) {
						$err = wfMessage('tv_pe_proto',
						  $link)->plain();
					}
					$specArr[$type] = $link;
				} else /* $n == 3 || $comma */ {
					# can rely on default empty
					//if ($comma) $specArr['first'] = '';
					if (!$comma) {
						if (substr($lineRem,0,1)==','){
							$key = 'first';
							$comma = true;
						} else	$key = 'subcontext';
						$specArr[$key] = trim($tmp);
						if (!$comma) $specArr['type'] =
						  'subcontext';
					}
					if ($comma) {
						$lineRem = substr($lineRem, 1);
						$n = strspn($lineRem, '[');
						if ($n != 3 && $n != 0) {
							$err = wfMessage('tv_pe_'.
							  'badobrcntrange', $n)
							  ->plain();
						} else {
							$specArr['type'] =
							  'range';
							if ($n == 0) {
							} else if (($pos=strpos(
							    $lineRem, ']]]', 3))
							  !== false) {
								$specArr['last']
								  = substr(
								    $lineRem, 3,
								    $pos - 3);
								$lineRem=ltrim(
								  substr(
								    $lineRem,
								    $pos + 3));
							} else	$err = wfMessage(
							  'tv_pe_unclosedrgbr')
							  ->plain();
						}
					}
				}
			}
		}
		if ($msgKey && $err) {
			$err = wfMessage('tv_pe_replmsg', $msgKey, $lineRem)
			         ->plain().$err;
		} else if ($msgKey) {
			$specArr['msgkey'] = $msgKey;
		}
		return array($res ? $res : $lineRem, $err, $specArr);
	}

	function validateExtUrl($url) {
		global $wgUrlProtocols;
		$re = function_exists('wfUrlProtocols') ?
		  wfUrlProtocols() : $wgUrlProtocols;
		return preg_match("/^($re)/", $url);
	}

	/**
	 * @todo add a parameter indicating whether the line containing the
	 * directive was ignored completely; adjust messages to fit.
	 */
	function addParseErr($instrLine,$errMsg,$introMsgKey='tv_parseerrmsg') {
		$lineNum = $instrLine['linenum'] - $this->mFirstLineNum + 1;
		$this->mParseErrors .= wfMessage($introMsgKey, $lineNum, $errMsg,
		  $instrLine['stars'], $instrLine['line'])->plain()."\n";
		$this->mParseErrors_Html .= wfMessage($introMsgKey)->rawParams(
		  $lineNum, $errMsg, $instrLine['stars'], '<code>'.
		  $instrLine['line'].'</code>')->escaped()."<br />\n";
	}

	function addParseWarning($instrLine, $warnMsg) {
		$this->addParseErr($instrLine, $warnMsg, 'tv_parsewarnmsg');
	}

	/**#@- */ # Temporarily escape the private block
	/**
	 * Converts the hierarchy config page at $wgHierarchyPage into a
	 * virtual hierarchy with nodes indicating where to import and prune
	 * contexts from the wiki's article title namespaces; stores the
	 * hierarchy in $this->mHierarchy.  Generates and validates the
	 * remappings list and stores it in $this->mRemappings. Generates the
	 * (somewhat) minimal SQL condition to query for imports and stores
	 * that in $this->mParseSql.  Sets $this->mParseErrors_Html and
	 * $this->mParseErrors in the event of parse errors/warnings.
	 * @param boolean $forceReparse Ignore the result of any previous parse
	 *  during this object's lifetime and force a reparse.
	 * @return boolean True if no warnings or errors; false otherwise.
	 * @access public
	 */
	function parseConfigPage($forceReparse = false) {
		/* Use a past result if possible. */
		if ($this->mParseResult !== null && !$forceReparse) {
			return $this->mParseResult;
		}

		# Localised for readability
		$specifiers = array(
			'NODE' => array('url', 'title', 'subcontext'),
			'CATEGORY' => array('title'),
			'IMPORT' => array('title'),
			'GRAFT' => array('title'),
			'PRUNE' => array('subcontext'),
			'BLOCK' => array('range'),
		);
		$parseRules = $this->mParseRules;
		$unspecifiableDirectives = array('ENDIMPORT');
		$allDirectives = array_merge($unspecifiableDirectives,
		  array_keys($specifiers));

		$fromStore = $this->readConfigPage($instructions, $remappings,
		  $firstLineNum);
		$this->mFirstLineNum = $firstLineNum;
		$this->validateRemappings($remappings, $fromStore);
		$this->mHierarchy = array(
			'ROOT' => true,
			'type' => 'ROOT',
			'id' => TV_ROOT_ID,
			'obj_version' => HIERARCHY_OBJ_VERSION,
			'generated_timestamp' => wfTimestampNow(TS_MW),
			'expanded' => true,
			'message_deps' => array(),
			'nodes' => array(),
		);
		$n =& $this->mHierarchy;
		$messageDeps =& $n['message_deps'];
		$elseHashDepth = -1;
		$elseGroupNum = 0;
		$sql = '';
		/** @todo Use $cStack further on in this function: its purpose
		  * is to allow for PRUNEing a CATEGORY, as-yet unimplemented
		  * here although it is supported (untested) within
		  * processCategory_r(). */
		$cStack/*Category stack*/ = $iStack/*Import stack*/ = $stack =
		  array();
		foreach ($instructions as $instrLine) {
			$newMsgs = array();
			$lineRem = $instrLine['line'];
			$arr = preg_split('/(\(|\s)/',
			  $lineRem, 2, PREG_SPLIT_DELIM_CAPTURE);
//			list($tmp, $chspl, $tmp2) = $arr;
			$tmp = $arr[0];
			$chspl = isset($arr[1]) ? $arr[1] : '';
			$tmp2 = isset($arr[2]) ? $arr[2] : '';
			$directive = strtoupper(trim($tmp));
			if (in_array($directive, $unspecifiableDirectives)) {
				$lineRem = $tmp2;
			} else if (!$directive && $directive == trim($tmp2)) {
				# diagnose no error for null directives - their
				# use is for ending if-else chains
				$directive = 'NULL';
			} else {
				list($lineRem, $err, $specArr) =
				  $this->parseSpecifier($lineRem);
				if ($err) {
					$this->addParseErr($instrLine,
					  wfMessage('tv_pe_specerr', $err)
					  ->plain());
					continue;
				} else if (isset($specArr['msgkey']) &&
				  $specArr['msgkey'] != '') {
					$newMsgs[$specArr['msgkey']] = true;
				}
				/* Over-verbose to avoid E_NOTICE warnings. */
				$splits = preg_split('/(\(|\s)/', $lineRem, 2,
				  PREG_SPLIT_DELIM_CAPTURE);
				$tmp = array_shift($splits);
				$chspl = (string)array_shift($splits);
				$lineRem = (string)array_shift($splits);
				$directive = strtoupper(trim($tmp));
				if (!in_array($directive, $allDirectives)) {
					$this->addParseWarning($instrLine,
					  wfMessage('tv_pe_implicitnodedir',
					    $directive)->plain());
					$directive = 'NODE';
					$lineRem = $tmp.$chspl.$lineRem;
					$chspl = '';
				}
				if (!in_array($specArr['type'],
				  $specifiers[$directive])) {
					$this->addParseErr($instrLine, wfMessage(
					  'tv_pe_dirspecmismatch', $directive,
					  $specArr['type'])->plain());
					continue;
				}
			}
			$lineRem = ltrim($lineRem);
			$title = $specArr[$specArr['type']];
			$type2 = $node2 = null;
			if ($directive == 'NODE') {
				if ($iStack) {
					$type = $specArr['type']=='subcontext' ?
					  'antiprune' : 'node';
				} else if ($specArr['type'] == 'subcontext') {
					# This is related to the diagnostic
					# 'tv_pe_scasfullnddesc' further down
					$this->addParseErr($instrLine,
					  wfMessage('tv_pe_scnotinimport',
					    wfMessage('tv_scnode'),
					    $specArr['dispname'], $title)
					    ->plain());
					continue;
				} else if (0) { /** @todo Add checks for the new
				                  * CATEGORY type */
				} else	$type = 'node';
			} else if (($directive=='PRUNE' || $directive=='BLOCK')
			  && !$iStack && !$cStack) {
				# This is related to the diagnostic
				# 'tv_pe_scasfullnddesc' further down
				$this->addParseErr($instrLine, wfMessage(
				  'tv_pe_scnotinimport', wfMessage(
				   $directive=='BLOCK'?'tv_block':'tv_scprune')
				   ->plain(), $specArr['dispname'],
				   $title)->plain());
				continue;
			} else if ($directive == 'GRAFT') {
				$type = 'node';
				$type2 = 'import';
			} else { /* IMPORT or CATEGORY */
				if ($directive == 'CATEGORY' && $cStack) {
					$type = 'antiprune';
				}
				$type = strtolower($directive);
			}

			if ($chspl != '(' && substr($lineRem, 0, 1) == '(') {
				$chspl = '(';
				$lineRem = ltrim(substr($lineRem, 1));
			}
			if ($chspl == '(') {
				# assumes that $type2 will not be conditionable
				# when $type is not conditionable
				if (!$parseRules[$type]['conditionable']) {
					$this->addParseErr($instrLine, wfMessage(
					  'tv_pe_condnotallowed', $directive)
					  ->plain());
					continue;
				}
				list($lineRem, $err, $conditions) =
				  $this->parseConditions($lineRem);
				if ($err) {
					$this->addParseErr($instrLine, wfMessage(
					  'tv_pe_dirconderr',$directive,$err)
					  ->plain());
					continue;
				}
			} else	$conditions = array();

			if ($type == 'prune' || $type == 'antiprune') {
				# store subcontext as though full title - this
				# partly complements the rtrim in queryDB()
				if (($t = Title::newFromText($title))) {
					$title = $t->getPrefixedText();
				}
			}

			$depth = count($stack);
			# All directives that fall within the scope of an import
			# are added as its children, even though they might be
			# at the same depth as the preceding import, so the
			# depth of each directive within the import's (possibly
			# nested) scope is artificially increased by one.
			$newDepth = $instrLine['depth'] + count($iStack);
			$iPopped = array();
			while ($iStack) {
				$z = count($iStack) - 1;
				if ($instrLine['depth'] < $iStack[$z] ||
				  $instrLine['depth'] == $iStack[$z] &&
				  (in_array($directive,
				     array('IMPORT', 'ENDIMPORT')))) {
					$newDepth--;
					array_unshift($iPopped,
					  array_pop($iStack));
				} else	break;
			}
			if ($newDepth > $depth) $parentType = $n['type'];
			else {
				$parentDepth = $newDepth - 1;
				$parentType = $parentDepth ?
				  $stack[$parentDepth]['type'] : 'ROOT';
			}
			if ($parentType == 'import' && $type == 'category') {
				$newDepth--;
				array_unshift($iPopped, array_pop($iStack));
			}

			$err = '';
			if (($type == 'prune' || $type == 'antiprune' ||
			  $type == 'block') && ($parentType == 'node' ||
			  $parentType == 'ROOT')) {
				$err = wfMessage('tv_pe_scasfullnddesc', wfMessage(
				 $type=='block'?'tv_block':'tv_pruneorscnode')
				 ->plain())->plain();
			} else if ($type == 'import' && $parentType != 'node' &&
			  $parentType != 'ROOT') {
				$err = wfMessage('tv_pe_misplimport', $title,
				  $parentType)->plain();
			} else if ($newDepth > $depth + 1) {
				# This test is performed last lest it match
				# 'import's that the test above provides a more
				# specific diagnostic for.
				$err = wfMessage('tv_pe_missingdepths',
				  $specArr['dispname'], $title, (int)$newDepth,
				  (int)$depth)->plain();
			}

			if ($err) {
				$this->addParseErr($instrLine, $err);
				# $iPopped will be non-empty where e.g. there
				# are one or more IMPORTs below a top-level node
				# (possibly nested as e.g * NODE ** IMPORT
				# *** NODE *** IMPORT) and then a top-level
				# (invalid because there's no preceding top-
				# level IMPORT) PRUNE.
				$iStack += $iPopped;
				continue;
			}

			if ($type == 'endimport') continue;

			$node = array(
				'type'	=> $type,
				'nodes'	=> array()
			);
			if ($node['type'] == 'block') {
				$node['first'] = $specArr['first'];
				$node['last'] = $specArr['last'];
			} else {
				if ($type=='node' && $specArr['type']=='url') {
					$node['ext'] = true;
					$key = 'url';
				} else	$key = 'title';
				$node['dispname'] = isset($specArr['dispname'])?
				  $specArr['dispname'] : '';
				$node[$key] = $title;
				if ($directive == 'CATEGORY') {
					$prefix = 'category:';
					if (strtolower(substr($title, 0,
					  strlen($prefix))) != $prefix)
						$node[$key] = ucfirst(
						  $prefix.$title);
				}
			}

			$c = $instrLine['stars'][strlen($instrLine['stars'])-1];
			$isHashNode = ($c == '#' && $type == 'node' &&
			  $specArr['type'] != 'subcontext');
			$restart = $isHashNode && ($newDepth >  $elseHashDepth);
			$same    = $isHashNode && ($newDepth == $elseHashDepth);
			if        ($restart) {
			                     $elseHashDepth    = $newDepth;
			                     $elseGroupNum++;
			} else if ($same   ) $node['elseflag'] = true;
			else                 $elseHashDepth    = -1;
			if ($type == 'null') continue;

			if ($conditions) $node['conditions'] = $conditions;

			$nsql = '';
			if ($node['type'] == 'import') {
				$nsql = $this->normaliseParsedImport($node);
				$iStack[] = $instrLine['depth'];
			}

			# Handle GRAFT parameters as described in
			# Help.Treeview_skin.Syntax.wiki
			$pKeys1 = $parseRules[$type]['parameters'];
			$pKeys2 = $type2 ?
			  $parseRules[$type2]['parameters'] : array();
			$err = $this->parseParameters($lineRem, $node, $newMsgs,
			  array_merge($pKeys1, $pKeys2));
			if ($err) $this->addParseErr($instrLine, wfMessage(
			  'tv_pe_parameters', $err)->plain());
			if (isset($node['id'])) {
				# Relies on a one-based not zero-based linenum
				if (isset($this->mNodeIdsParsed[$node['id']])) {
					list($oLineNum, $oGrp) =
					  $this->mNodeIdsParsed[$node['id']];
					if ($oGrp && $elseGroupNum == $oGrp) {
						/* Avoid warnings for nodes
						 * that are part of the same
						 * if-else group. */
					} else {
						$this->addParseWarning(
						  $instrLine,
						  wfMessage('tv_pe_repeatedid',
						   $node['id'], $oLineNum -
						   $this->mFirstLineNum + 1)
						   ->plain());
						unset($node['id']);
					}
				} else	$this->mNodeIdsParsed[$node['id']] =
				  array($instrLine['linenum'], $elseGroupNum ?
				    $elseGroupNum : 0);
			}
			if ($type2) {
				$node2 = $node;
				$node2['type'] = $type2;
				if ($conditions &&
				  !$parseRules[$type2]['conditionable']) {
					unset($node2['conditions']);
				}
				$pKeys2Excl = array_diff($pKeys2, $pKeys1);
				$pKeysBoth = array_intersect($pKeys1, $pKeys2);
				foreach ($pKeys2Excl as $k)
					if(isset($node [$k])) unset($node [$k]);
				foreach ($pKeysBoth  as $k)
					if(isset($node2[$k])) unset($node2[$k]);
				if ($node2['type'] == 'import') {
					$nsql = $this->normaliseParsedImport(
					  $node2);
					$iStack[] = $instrLine['depth'] + 1;
				}
			}

			# Optimisation opportunity: refine the sql so that
			# results that will later be pruned are not returned
			if ($nsql) {
				if ($sql) $sql .= ' OR ';
				$sql .= $nsql;
			}

			if ($newDepth <= $depth) {
				if ($newDepth < $depth) {
					for ($i = $depth - $newDepth; $i; $i--){
						array_pop($stack);
					}
				}
				$idx = count($stack[$newDepth - 1]['nodes']);
				$stack[$newDepth - 1]['nodes'][$idx] = $node;
				$n =& $stack[$newDepth - 1]['nodes'][$idx];
			} else if ($newDepth == $depth + 1) {
				# entry-point for first node
				$n['nodes'][] = $node;
				$stack[] =& $n;
				$n =& $n['nodes'][count($n['nodes']) - 1];
			} else {
				wfDebug(__METHOD__.": this should be ".
				  "unreachable\n");
				continue; # unchecked level-skip
			}
			if ($node2) {
				$n['nodes'][] = $node2;
				$stack[] =& $n;
				$n =& $n['nodes'][count($n['nodes']) - 1];
			}
			$messageDeps = array_merge($messageDeps, $newMsgs);
		}
		if ($this->mParseErrors) {
			wfDebug(__METHOD__.": ".wfMessage('tv_parseerrintro')
			  ->plain().": \n".$this->mParseErrors);
		}
		$this->mParseSql = $sql;
		$this->mParseResult = $this->mParseErrors ? false : true;
		return $this->mParseResult; /* Can return immediately too. */
	}

	/**#@+ @access private */ # Return to the private block
	/**
	 * Normalises the 'title' component of a just-parsed import node and
	 * builds the sql condition to select the import.
	 * @return String SQL condition (or undefined).
	 */
	function normaliseParsedImport(&$node) {
		if ($node['title'] == '' || $node['title'] == '_') {
			$node['title'] = '_';
			$ret = '(page_namespace='.NS_MAIN.')';
		} else $ret = '';
		if (!$ret && strpos($node['title'], ':') === false) {
			$nt = $node['title'].':NULL';
			$t = Title::newFromText($nt);
			# convert e.g. Project to clc-wiki
			if ($t && $t->getDBkey() == 'NULL') {
				$node['title'] = substr($t->getPrefixedText(),
				  0, -strlen(':NULL'));
				$ret='(page_namespace='.$t->getNamespace().')';
			}
		}
		if (!$ret && ($t = Title::newFromText($node['title']))) {
			$ret = '(page_namespace='.$t->getNamespace().
			  " AND (page_title LIKE '".addslashes($t->getDBkey()).
			  ":%' OR page_title='".$t->getDBkey()."'))";
			$a = explode(':', addslashes($t->getPrefixedText()));
			# this is the other complement to the rtrim in queryDB()
			foreach ($a as $k => $v) $a[$k] = rtrim($v);
			$node['title'] = implode(':', $a);
		}
		return $ret;
	}

	/** Precondition: $lineRem has been ltrim()ed. */
	function parseParameters($lineRem, &$node, &$usedMsgs, $paramKeys) {
		$pRules = $this->mParseRules;
		$err = false;
		while ($lineRem) {
			$param = whichstarts($lineRem, $paramKeys);
			if (!$param) {
				$err = wfMessage('tv_pe_invalidparam',
				  $node['type'], $lineRem)->plain();
				break;
			}
			/* Don't ltrim() in case splitter starts with w/s and so
			 * that missing whitespace can be warned about for
			 * unsplit params. */
			$lineRem = substr($lineRem, strlen($param));
			$po = $this->mParameterOptions[$param];
			$x = isset($po['splitter']) ? $po['splitter'] : false;
			$key = isset($po['key']) && $po['key'] != '' ?
			  $po['key'] : $param;
			if ($x) {
				$pos = strpos($lineRem, $x);
				if ($pos === false ||
				  trim(substr($lineRem, 0, $pos))) {
					$err = wfMessage('tv_pe_nosplit', $x,
					  $param)->plain();
					break;
				}
				$lineRem = ltrim(substr($lineRem,
				  $pos + strlen($x)));

				/* Special-case handling to parse ICONS_PER_NODE
				 * (optional) icons. */
				if ($param == 'icons') {
					$icon1 = '\[\[([^\]]*?):([^\]]*?)\]\]';
					$icon = "(($icon1){0,1})";
					$sep = '(\s*),(\s*)';
					$opticon = "(($sep$icon){0,1})";
					$opticons2="(($sep$icon$opticon){0,1})";
					$re = "/$icon$opticons2(\s*)(.*)/";
					if (!preg_match($re,$lineRem,$matches)){
						$err = wfMessage('tv_pe_iconsmalfm',
						  $lineRem)->plain();
						break;
					}
					$node[$key] = array($matches[4],
					  $matches[12], $matches[20]);
					$lineRem = $matches[22];
					continue;
				}

				$qchar = isset($po['quotechar']) ?
				  $po['quotechar'] : '';
				$vals = isset($po['values']) ?
				  $po['values'] : array();
				$val = whichstarts($lineRem, $vals);
				if ($val !== null) {
					$node[$key] = $val;
					$lineRem=substr($lineRem,strlen($val));
				} else if ($qchar &&
				  substr($lineRem,0,strlen($qchar)) == $qchar) {
					list($unesc, $lineRem) = $this->
					  parseEscapeable($qchar, $lineRem);
					if ($unesc === false) {
						$err = wfMessage('tv_pe_unclosedqt',
						  $qchar, $param)->plain();
						break;
					} else $node[$key] = $unesc;
				} else if ($qchar && empty($po['msgkeyable'])) {
					$err = wfMessage('tv_pe_nostartqt', $qchar,
					  $param)->plain();
					break;
				}
				if (!isset($node[$key])) {
					list($val, $lineRem) =
					  xsplit2(' ', $lineRem);
					if (isset($po['msgkeyable']) &&
					  $po['msgkeyable'] == true) {
						$node[$key] = $this->configMsgFunc($val);
						$usedMsgs[$val] = true;
					} else	$node[$key] = $val;
				}
			} else {
				$lineRem1 = $lineRem;
				$potErr = 'tv_pe_warnwsmiss';
				if (isset($po['conditionable']) &&
				  $po['conditionable'] == true) {
					$potErr = 'tv_pe_warnnowsopparen';
					$lineRem = ltrim($lineRem);
					if ($lineRem[0] == '(') {
						$lineRem1 = ' '; /*Avoid $err*/
						list($lineRem,$err,$conditions)=
						  $this->parseConditions(
						    substr($lineRem, 1));
						if ($err) $err = wfMessage(
						  'tv_pe_dirconderr', $param,
						  $err)->plain();
						else $node[$key] = $conditions;
					} else	$node[$key] = array(array(
					  CD_ALWAYS, false));
				} else	$node[$key] = true;
				if ($lineRem1 && $lineRem1[0] != ' ') {
					$err = wfMessage($potErr, $param)
					  ->plain();
				}
			}
			$lineRem = ltrim($lineRem);
		}
		return $err;
	}

	/** Parses the string in $lineRem and unescapes doubled $escSeq tokens
	  * by undoubling them.  If $quoted is true, then $escSeq also
	  * represents the start and end quote-character of a quoted string; in
	  * that case $lineRem must start with $escSeq, being the opening quote,
	  * and if the closing quote is not part of $lineRem then the first
	  * element of the returned array will be false.
	  * If $quoted is instead false, then all doubled $escSeq tokens in
	  * $lineRem are undoubled, without expectation that $lineRem starts
	  * with any particular character.
	  * @param string $escSeq
	  * @param string $lineRem
	  * @param boolean $quoted
	  * @return Array of two elements:
	  *  [0] => the unescaped string or false if the string is unterminated.
	  *  [1] => if $quoted is true: the portion of $lineRem subsequent to
	  *           the terminating quote, or the passed-in $lineRem if the
	  *           string was unterminated;
	  *         if $quoted is false: true if the final character of $lineRem
	  *           was an undoubled $escSeq; otherwise false (unimplemented).
	  */
	function parseEscapeable($escSeq, $lineRem, $quoted = true) {
		$lineRem_in = $lineRem;
		$seqLen = strlen($escSeq);
		$done = $quoted ? false : 0;
		$unesc = '';
		while (!$done) {
			if ($done === false) $lineRem=substr($lineRem, $seqLen);
			$pos = strpos($lineRem, $escSeq);
			if ($pos === false) break;
			else $done = false;
			$unesc .= substr($lineRem, 0, $pos);
			$lineRem = substr($lineRem, $pos + $seqLen);
			if (substr($lineRem, 0, $seqLen) == $escSeq) {
				$unesc .= $escSeq;
			} else	$done = true;
		}
		if (!$quoted) $unesc .= $lineRem;
		else if ($pos === false) {
			$unesc = false;
			$lineRem = $lineRem_in;
		}
		return array($unesc, $quoted ?
		  $lineRem : ($lineRem == '' && $done === false));
	}

	/** Replaces Treeview-recognised variable tokens in the given text.
	  * @return Array. A list: the first element is the post-replacement
	  * text and the second is a map of the values of replacements made,
	  * keyed by replacement token.
	  */
	function rvars($text) {
		$ret = $text;
		$repls = array();
		$replaceMap = $this->mState->getReplacements();
		foreach ($replaceMap as $before => $after) {
			$new = str_replace($before, (string)$after, $ret);
			if ($ret != $new) $repls[$before] = $after;
			$ret = $new;
		}
		return array($ret, $repls);
	}

	/**
	 * Queries the database for article titles given the condition
	 * $this->mParseSql which is generated by readConfigPage().  Constructs
	 * a hierarchy based on contexts being separated by colons, recognising
	 * namespaces as context-starters.  Stores that hierarchy in
	 * $this->mHierarchyDB.
	 */
	function queryDB() {
		$sqlCond = $this->mParseSql;
		// $dbw = wfGetDB(DB_MASTER);
		$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
		$pagetbl = $dbw->tableName('page');
		$this->mHierarchyDB = array();
		# exclude talk namespaces assuming that they're all odd numbered
		$sql = "
			SELECT	page_namespace as ns,
			        page_id as id,
				page_title as ttl
			FROM	$pagetbl
			WHERE	($sqlCond) AND
				page_namespace % 2 <> 1";
		if ($sqlCond && ($res = $dbw->query($sql))) {
			while ($obj = $res->fetchObject()) {
				# Optimisation opportunity: a small
				# inefficiency is that this title object is
				# recreated in realiseNodeLink() unless this
				# node has been pruned (the fact that pruned
				# nodes are returned here at all presents an
				# optimisation possibility itself as listed
				# in parseConfigPage().
				$t = Title::makeTitle($obj->ns, $obj->ttl);
				$contexts = explode(':', $t->getPrefixedText());
				$wb =& $this->mHierarchyDB;
				$is_ns = ($obj->ns != 0);
				for ($i = 0; $i < count($contexts); $i++) {
					$subC = $contexts[$i];
					$id = $i+1==count($contexts)?$obj->id:0;
					# remove trailing spaces (originally
					# underscores) from each context
					# component since X___:X is unconverted
					# by the Title class but X___ alone
					# is converted to X
					$subC = rtrim($subC);
					if (!isset($wb[$subC]) ||
					  count($wb[$subC]) == 0) {
						$wb[$subC]['nodes'] = array();
						if ($id) $wb[$subC]['aid']=$id;
						if ($is_ns !== -1) {
							$wb[$subC][$is_ns ?
							'nsflag':'pgflag']=true;
						}
					}
					# deal with pages in the main namespace
					# that have the same name as a namespace
					# N.B. 'pgflag' is only set for
					# top-level (i.e. no colons) articles
					if ($is_ns !== -1) {
						$wb[$subC][$is_ns ? 'nsflag' :
						  'pgflag'] = true;
					}
					$wb =& $wb[$subC]['nodes'];
					$is_ns = -1;
				}
			}
		}
	}

	/**
	 * Populates $this->mContextNodeIds and $this->mNodeIdContexts from the
	 * DB; begins a locking transaction that's committed in saveNewNodeIDs()
	 */
	function queryUsedNodeIDs() {
		// $dbw =& wfGetDB(DB_MASTER);
		$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
		$dbw->begin();
		$this->mContextNodeIds = $this->mNodeIdContexts = array();
		if (($res = $dbw->select('treeview_used_nodeids', array(
		  'tv_nodeid','tv_visiblecontext'), '', 'FOR UPDATE',
		   __METHOD__))) {
			$count = $dbw->numRows($res);
			while ($obj = $res->fetchObject()) {
				$nid = $obj->tv_nodeid;
				$vCtx = $obj->tv_visiblecontext;
				# user-specified IDs take precedence and can
				# change visible context, whereas IDs stored to
				# the database are assumed to always occur in
				# the same visible context.
				if (isset($this->mNodeIdsParsed[$nid])) {
					continue;
				}
				$this->mNodeIdContexts[$nid] = $vCtx;
				$this->mContextNodeIds[$vCtx]['pastids'][]=$nid;
			}
		}
		wfDebug(__METHOD__.": QUERIED '$count' previously used node ".
		  "ids from the database and LOCKED treeview_used_nodeids\n");
	}

	/**
	 * Saves $this->mNewNodeIds to the database; commits the locking
	 * transaction that was opened in queryUsedNodeIDs().
	 */
	function saveNewNodeIDs() {
		wfDebug(__METHOD__.": INSERTING ".count($this->mNewNodeIds).
		  " new node id(s) into the db; UNLOCKING ".
		  "treeview_used_nodeids.\n");
		// $dbw =& wfGetDB(DB_MASTER);
		$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
		if ($this->mNewNodeIds) {
			$dbw->insert('treeview_used_nodeids',
			  $this->mNewNodeIds, 'Hierarchy::saveNewNodeIDs');
		}
		$dbw->commit();
	}

	function recurseHierarchy($nodeFunc, $hierarchy = null,
	  $isExtFunc = false, $funcData = null) {
		if (!$hierarchy) $hierarchy =& $this->mHierarchy;
		return Hierarchy::recurseHierarchy_r($nodeFunc, $hierarchy,
		  $isExtFunc, $funcData);
	}

	/** @static */
	static function recurseHierarchy_r($nodeFunc, &$vn, $isExtFunc = false,
	  $funcData = null) {
		$ret = $isExtFunc ? $nodeFunc($vn, $funcData) :
		  Hierarchy::$nodeFunc($vn, $funcData);
		if ($ret) return $ret;
		for ($k = 0; $k < count($vn['nodes']); $k++) {
			$vnChild =& $vn['nodes'][$k];
			$ret = Hierarchy::recurseHierarchy_r($nodeFunc,
			  $vnChild, $isExtFunc, $funcData);
			if ($ret) return $ret;
		}
	}

	static function initialiseNodeState(&$vn) {
		if (count($vn['nodes']) <= 0) {
			$vn['state'] = TV_LEAF;
		} else if (!empty($vn['expanded'])) {
			$vn['state'] = TV_EXPANDED;
		} else	$vn['state'] = TV_COLLAPSED;
	}

	static function trimNodeForCache(&$vn) {
		unset($vn['title_obj']);
	}

	static function isImageInNode(&$vn, $imageName) {
		for ($i = 0; $i < ICONS_PER_NODE; $i++) {
			if (!isset($vn['icons'][$i])) continue;
			else if ($vn['icons'][$i] == $imageName) return true;
		}
		return false;
	}

	static function nodeIfIdMatch(&$vn, $nodeId) {
		return $vn['id'] == $nodeId ? $vn : false;
	}

	/**
	 * Performs recursive post-parse processing on the hierarchy.  The
	 * main tasks are to perform initial replacements, to set the id and
	 * display name of 'node's, to delegate the importing of IMPORTs and
	 * CATEGORYs and to realise the imported nodes of those directives.
	 * @param Mixed $vn A node in mHierarchy of type 'node' or 'ROOT'.
	 * @param string $visContext Colon-separated.  The parent context
	 * within which this node will be displayed (i.e. a context comprising
	 * $vn['dispname']s)
	 * @param string $prevId The ID that was set for the previous node, only
	 * required if $vn has its elseflag set.
	 */
	function processParseTree_r(&$vn, $visContext = '', $prevId = '') {
		# mnemonics: vn for Virtual hierarchy Node

		if ($vn['type'] != 'ROOT') {
			$this->doNodeReplacements($vn, TV_CONDREPL_BOTH, true);
			if (empty($vn['ext']) && $vn['title']) {
				# normalise the title and get its id for the
				# later buildHierarchyHtml_r() existence test -
				# called prior to caching mHierarchy
				if (($t = Title::newFromText($vn['title']))) {
					$vn['title'] = $t->getPrefixedText();
					$vn['aid'] = $t->getArticleID();
					$uExtra = isset($vn['urlextra']) ?
					  $vn['urlextra'] : '';
					$vn['url'] = $t->getLocalURL($uExtra);
					$vn['title_obj'] = $t;
				}
			}

			# The node's display name is not set if it's empty and
			# the title contains replacement variables - in that
			# case it's the task of the link functions to generate
			# a default ...
			if (empty($vn['dispname']) &&
			  empty($vn['title_prereplace'])) {
				if (empty($this->mRemappings[$vn['title']])) {
					$vn['dispname'] = $vn['title'];
				} else	$vn['dispname'] =
				  $this->mRemappings[$vn['title']];
			}
			# ...but in that case, the "visible context" is set to
			# the pre-replacement title so that $newVis has
			# a consistent non-empty value for the subcontext when
			# passed in to setNodeId()
			$newDispname = isset($vn['dispname_prereplace']) ?
			  $vn['dispname_prereplace'] :
			  (isset($vn['dispname']) ? $vn['dispname'] :
			    $vn['title_prereplace']);
			$newVis = $this->append($visContext, $newDispname);
			# A later node in an if-else group inherits the id of
			# the initial node unless the later's id is supplied
			if (!empty($vn['elseflag']) && $prevId &&
			  empty($vn['id'])) {
				$vn['id'] = $prevId;
			} else	$prevId = $this->setNodeId($vn, $newVis);
		} else	$newVis = '';
		$vb =& $vn['nodes'];
		$i = 0;
		while ($i < count($vb)) {
			if ($vb[$i]['type'] == 'import') {
				$i += $this->processImport($vn,$i,$newVis) - 1;
			} else if ($vb[$i]['type'] == 'node') {
				$this->processParseTree_r($vb[$i],$newVis);
			} else if ($vb[$i]['type'] == 'category') {
				if (!$this->processCategory($vb[$i], $newVis)) {
					array_splice($vb, $i--, 1);
				}
			} else {
				wfDebug(__METHOD__.": UNEXPECTED node type: ".
				  "'{$vb[$i]['type']}' (dispname: ".
				  "'{$vb[$i]['dispname']}')\n");
			}
			$i++;
		}
	}

	/** Appends a context to a colon-separated string of them. */
	function append($baseContext, $subContext) {
		return ($baseContext ? "$baseContext:" : '').$subContext;
	}

	/**
	 * Generates a guaranteed-unique id for the node if one is not already
	 * set (otherwise returns immediately), also guaranteed unique for all
	 * previous treeview instances that the setting was true for if the
	 * $wgStoreTreeviewNodeIdsInDB setting is true.  Caveat:
	 * no guarantee is made that nodes in the same branch that share a
	 * display name will not swap or change ids if their sort order changes
	 * or if one or more of them are later deleted.
	 * @param Mixed $vn A node within $this->mHierarchy
	 * @param string $fullVisContext The full visible context for the node
	 * as a colon-separated string, pre-replacements if there are any.
	 * @return string; the node's id, already set within the node.
	 * @note subsequent nodes in an if-else chain share the id of the
	 * first: the article id component will be misrepresentative if they
	 * have different titles (this is actually implemented in
	 * processParseTree_r()).  The article-id component will also be
	 * misrepresentative if a different title with the same node display
	 * text is later slotted into the same position in the treeview.
	 */
	function setNodeId(&$vn, $fullVisContext) {
		static /* const */ $baseCharsMax = 16;

		if (!empty($vn['id'])) return $vn['id'];

		# varchar columns in the database trim trailing spaces
		$fullVisContext = rtrim($fullVisContext);
		$newid = null;
		$ref =& $this->mContextNodeIds[$fullVisContext];
		if (isset($ref['pastids'])) {
			natsort_newkeys($ref['pastids']);
			$ref['usedids'][]=$newid=array_shift($ref['pastids']);
		} else if (count((array)$ref) > 0) {
			# false strrpos or non-integral substr result in $seq==0
			$last = array_pop($ref['usedids']);
			$ref['usedids'][] = $last;
			$pos = strrpos($last, '_');
			$seq = (int)substr($last, $pos + 1);
		} else	$seq = 0;
		if (!$newid) {
			$disp = isset($vn['dispname_prereplace']) ?
			  $vn['dispname_prereplace'] : $vn['dispname'];
			$dispfilt = filter_non_alnum($disp);
			$newid = '_'.substr($dispfilt, 0, $baseCharsMax);
			$newid = strtolower($newid).'_';
			# Don't use article id when the title contains
			# variables - both are subject to change.
			if (!empty($vn['aid']) &&
			     empty($vn['title_prereplace'])) {
				$newid .= $vn['aid'];
			}
			$baseid = $newid;
			do {
				$newid = $baseid.'_'.(++$seq);
			} while (isset($this->mNodeIdContexts[$newid]) ||
			  isset($this->mNodeIdsParsed[$newid]) ||
			  $newid == TV_ROOT_ID);
			$this->mNodeIdContexts[$newid] = $fullVisContext;
			$ref['usedids'][] = $newid;
			$this->mNewNodeIds[] = array(
				'tv_nodeid' => $newid,
				'tv_visiblecontext' => $fullVisContext
			);
		}
		$vn['id'] = $newid;
		return $newid; # warning: prior conditional return
	}

	/** Query the DB for the top-level category (might not exist if it has
	  * not yet been edited). Delegate recursive querying for further
	  * children. */
	function processCategory(&$vn, $visContext) {
		$ret = false;
		$this->setNodeId($vn, $visContext);
		// $dbw = wfGetDB(DB_MASTER);
		$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
		$page = $dbw->tableName('page');
		$title = $vn['title'];
		$prefix = 'category:';
		if (strtolower(substr($title, strlen($prefix))) == $prefix) {
			$title = substr($title, strlen($prefix));
			$vn['title'] = $title;
		}
		$t = Title::newFromText($title);
		if ($t) {
			$dbTitle = $t->getDBkey();
			$sql = "
				SELECT     page_title,
				           page_id,
				           page_namespace
				FROM       $page
				WHERE      page_namespace = ".NS_CATEGORY." AND
				           page_title = '".$dbw->strencode($dbTitle)."'";
			$res = $dbw->query($sql);
			$vn['dispname'] = $dbTitle;
			if ($res) {
				$ret = true;
				$obj = $res->fetchObject();
			}
			if ($res && $obj) {
				$vn['aid'] = $obj->page_id;
			}
			$seenCats = array();
			$seenCats[$dbTitle] = true;
			$this->processCategory_r($vn, $visContext.':'.
			  $t->getText(), $dbTitle, $seenCats, !(isset($vn['showpages']) && $vn['showpages']));
		}
		return $ret;
	}

	/** Recursively query for the sub-categories of a CATEGORY directive and
	  * add them to the virtual nodes structure. */
	function processCategory_r(&$vn, $visContext, $catTitle, &$seenCats,
	  $categoriesOnly, $pruned = false) {
		/* Note: the average number of queries could be reduced by
		 * using a batch approach for children rather than - as is done
		 * here - recursing on each child, but the extra complexity
		 * probably isn't worth it.  If DB contention proved a
		 * bottleneck then it might be worth investigating though. */
		$newNodes = array();
		$antiprune = false;
		// $dbw = wfGetDB(DB_MASTER);
		$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
		$categorylinks = $dbw->tableName('categorylinks');
		$page = $dbw->tableName('page');
		$sql = "
		SELECT     c.page_title AS title,
		           c.page_id,
		           c.page_namespace
		FROM       $categorylinks cl
		LEFT OUTER JOIN $page p
		ON         cl_to = p.page_title".($categoriesOnly ? " AND
		           p.page_namespace = ".NS_CATEGORY : '')."
		INNER JOIN $page c
		ON         cl_from = c.page_id".($categoriesOnly ? " AND
		           c.page_namespace = ".NS_CATEGORY : '')."
		WHERE      cl.cl_to = '".$dbw->strencode($catTitle)."'";
		if ($res = $dbw->query($sql)) {
			while ($obj = $res->fetchObject()) {
				$t = Title::makeTitle($obj->page_namespace, $obj->title);
				/* Identify and deal with any prunes or
				 * antiprunes. */
				$prune2 = 0;
				for ($i = 0; $i < count($vn['nodes']); $i++) {
					$t2 = Title::newFromText(
					  $vn['nodes'][$i]['title']);
					if ($t && $t2 &&
					  $t->getDBkey() == $t2->getDBkey()) {
						$prune2 = ($vn['nodes'][$i]
						  ['type'] == 'prune');
						$node = $vn['nodes'][$i];
						if (!$prune2) $antiprune = true;
						break;
					}
				}
				if ($prune2 === 0) {
					$node = array(
						'title'=>$t->getPrefixedText(),
						'dispname' => $t->getText(),
						'nodes' => array()
					);
				}
				$node['type'] = 'node';
				$node['aid'] = $obj->page_id;
				$visContext2 = $visContext.':'.$obj->title;
				if (empty($seenCats[$obj->title])) {
					$seenCats[$obj->title] = true;
					$tmp = $this->processCategory_r($node,
					  $visContext2, $obj->title, $seenCats,
					  $categoriesOnly, $prune2 ||
					  $pruned && $prune2 !== false);
					if ($tmp) $antiprune = true;
					if ($tmp || !$pruned) {
						$this->setNodeId($node,
						  $visContext2);
						$newNodes[] = $node;
					}
				}
			}
			$this->sortCatNodes($newNodes);
		}
		$vn['nodes'] = $newNodes;
		return $antiprune;
	}

	function sortCatNodes(&$nodes) {
		$sortArr = array();
		foreach ($nodes as $node) $sortArr[] = $node['dispname'];
		natsort($sortArr);
		$newNodes = array();
		foreach ($sortArr as $idx => $dispname) {
			$newNodes[] = $nodes[$idx];
		}
		$nodes = $newNodes;
	}

	/**
	 * Delegates the generation of the imported subtrees and splices those
	 * imported subtrees into the 'import' directive node's parent in place
	 * of that import directive node.  Can be indirectly called recursively.
	 *
	 * @param Mixed $vn The parent node containing the import node.
	 * @param integer $i The index of the import node within $vn['nodes'].
	 */
	function processImport(&$vn, $i, $visContext) {
		$this->doNodeReplacements($vn['nodes'][$i], TV_CONDREPL_BOTH,
		  true);
		$vb =& $vn['nodes'];
		if ($vb[$i]['title'] == '_' || $vb[$i]['title'] == '') {
			$prunens = true; # prune top-level namespace's
			$wb = $this->mHierarchyDB;
			$baseContext = '';
		} else {
			$prunens = false;
			$baseContext = $vb[$i]['title'];
			$wb = $this->findContextDBTree($baseContext);
		}
		$r = $this->mParseRules['import'];
		$params = x_array_key_values($vb[$i], $r['parameters']);
		$propagated = x_array_key_values($params, $r['propagated']);
		$this->mergeImportBranch_r($vb[$i]['nodes'], $wb,
		  $baseContext, $visContext, $params, $propagated, false,
		  $prunens);
		$cnt = count($vb[$i]['nodes']);
		array_splice($vb, $i, 1, $vb[$i]['nodes']);
		return $cnt;
	}

	/**
	 * Locates the branch of $this->mHierarchyDB corresponding to the full
	 * context $fullContext.  Expects that each subcontext of $fullContext
	 * has been rtrim()ed as in normaliseParsedImport().
	 * @param string $fullContext
	 */
	function findContextDBTree($fullContext) {
		$wb = $this->mHierarchyDB;
		$contexts = explode(':', $fullContext);
		$found = $contexts ? true : false;
		foreach ($contexts as $context) {
			if (!isset($wb[$context]['nodes'])) {
				$found = false;
				break;
			} else	$wb = $wb[$context]['nodes'];
		}
		return $found ? $wb : array();
	}

	/**
	 * Merges a parse-tree-derived (mHierarchy) branch with a colon-
	 * separated-wiki-titles-in-database-derived (mHierarchyDB) branch,
	 * performing sorting and resolution of prunes, antiprunes and blocks.
	 * $vb could be a branch off a prune or antiprune directive node, or
	 * directly off an import directive node.
	 *
	 * Expectations:
	 *  If $propagated is non-empty then its values are identical to those
	 *   of $params for shared keys.
	 *  The ['type'] of each of $vb's ['nodes'] is one of 'node', 'prune',
	 *   'antiprune' or 'block' and is NOT 'import'.
	 */
	function mergeImportBranch_r(&$vb, $wb, $baseContext, $visContext,
	  $params, $propagated, $prune = false, $prunens = false) {

		# propagated IMPORT parameters that aren't used later needn't
		# take up space in the node
		static $parmsNoStore = array('merge');
		$parmsProp = $this->mParseRules['import']['propagated'];
		$storeKeys = array_diff($parmsProp, $parmsNoStore);
		$paramsToStore = x_array_key_values($params, $storeKeys);

		$this->performRemappings($vb, $baseContext);
		if (!isset($params['merge']) || $params['merge'] == 'sort') {
			$effectiveOrder = 'prepend';
		} else	$effectiveOrder = $params['merge'];
		if ($effectiveOrder == 'append') {
			$vb[] = array(
				'type' => 'block',
				'first' => '',
				'last' => '',
				'exclude' => 'first'
			);
		}

		# Don't import namespaces other than the primary namespace when
		# $prunens is true
		$noRecurses = array();
		$wbSubCs = array_keys($wb);
		if ($prunens) {
			$i = 0;
			while ($i < count($wbSubCs)) {
				if (!empty($wb[$wbSubCs[$i]]['nsflag']) &&
				     empty($wb[$wbSubCs[$i]]['pgflag'])) {
					array_splice($wbSubCs, $i, 1);
				} else {
					if(!empty($wb[$wbSubCs[$i]]['nsflag'])){
						$noRecurses[] = $wbSubCs[$i];
					}
					$i++;
				}
			}
		}

		$wbRemapToSubC = $this->remapSubs($wbSubCs, $baseContext);
		$wbRemappedSCs = $wbSubCs;
		natsort_newkeys($wbRemappedSCs);

		$antiprune = false;
		$i = 0;
		while ($i < count($vb)) {
			$vn =& $vb[$i];
			if ($vn['type']=='antiprune' || $vn['type']=='prune') {
				$noRecurse=in_array($vn['title'], $noRecurses);
				$newParamsToStore = array_merge(
				  x_array_key_values($propagated, $storeKeys),
				  x_array_key_values($vn, $storeKeys));
				$subContext = $vn['title'];
				$found = array_keys($wbRemapToSubC,$subContext);
				/* values are unique */
				$mapping = isset($found[0]) ? $found[0] :
				  $subContext;
				$newNode = $this->generateImportedNode(
				  $newParamsToStore, $baseContext, $subContext,
				  $mapping, $wb, $vn['type']);
				$newVis = $this->append($visContext,
				  $newNode['dispname']);
				$new_wb = isset($wb[$subContext]['nodes']) ?
				  $wb[$subContext]['nodes'] : array();
				if ($vn['type'] == 'antiprune') {
					$this->setNodeId($newNode, $newVis);
				}
				if (($found || count($vn['nodes']) > 0) &&
				  !$noRecurse) {
					$pkeys = $this->mParseRules[$vn['type']]
					  ['parameters'];
					$newParams = array_merge($propagated,
					  x_array_key_values($vn, $pkeys));
					$newNode['nodes'] = $vn['nodes'];
					$res = $this->mergeImportBranch_r(
					  $newNode['nodes'], $new_wb,
					  $newNode['title'], $newVis,
					  $newParams, $propagated,
					  $prune || $vn['type'] == 'prune');
				} else	$res = false;
				if ($vn['type'] == 'antiprune' || $res) {
					if ($res) $this->setNodeId($newNode,
					  $newVis);
					$antiprune = true;
					$vn = array_merge($vn, $newNode);
					$this->doNodeReplacements($vn,
					  TV_CONDREPL_BOTH, true);
				} else {
					array_splice($vb, $i, 1);
					$i--;
				}
				if ($found) {
					$n = array_search($mapping,
					  $wbRemappedSCs);
					array_splice($wbRemappedSCs, $n, 1);
					unset($wbRemapToSubC[$mapping]);
				}
			} else if ($vn['type'] == 'node') {
				$antiprune = true;
				$this->processParseTree_r($vn, $visContext);
			}
			$i++;
		}

		if ($prune && !$antiprune) return false;

		$ignore = array('importednode', 'node', 'prune', 'antiprune', 'category');
		$i = 0;
		while ($i < count($vb)) {
			$vn =& $vb[$i];
			if ($vn['type'] == 'block') {
				$nodes = $this->processBlock($vn, $wb,
				  $wbRemappedSCs, $wbRemapToSubC, $baseContext,
				  $visContext, $paramsToStore, $propagated,
				  !empty($params['pruneleafs']), $noRecurses);
				array_splice($vb, $i, 1, $nodes);
				$i += count($nodes) - 1;
			} else if (!in_array($vn['type'], $ignore)) {
				wfDebug(__METHOD__.": encountered node of ".
				  "UNEXPECTED type: '{$vn['type']}'\n");
			}
			$i++;
		}

		if ($effectiveOrder == 'prepend') {
			# relying on unset == '' for array members
			$nodes = $this->processBlock(array('type'=>'block'),
			  $wb, $wbRemappedSCs, $wbRemapToSubC, $baseContext,
			  $visContext, $paramsToStore, $propagated,
			  !empty($params['pruneleafs']), $noRecurses);
			array_splice($vb, 0, 0, $nodes);
		}

		if (isset($params['merge']) && $params['merge'] == 'sort') {
			$this->sortImportBranch($vb);
		}

		return $antiprune; # warning: prior conditional early exit
	}

	function performRemappings(&$vb, $baseContext) {
		for ($i = 0; $i < count($vb); $i++) {
			$vn =& $vb[$i];
			if ($vn['type']=='antiprune' || $vn['type']=='prune') {
				$full=$this->append($baseContext,$vn['title']);
				$vn['dispname'] =
				  isset($this->mRemappings[$full]) ?
				  $this->mRemappings[$full] : $vn['title'];
			}
		}
	}

	/**
	 * Converts each entry in $wbSubCs to its remapping (no change if
	 * no remapping exists); returns a reverse-remapping array.
	 */
	function remapSubs(&$wbSubCs, $baseContext) {
		$wbRemapToSubC = array();
		for ($i = 0; $i < count($wbSubCs); $i++) {
			$subContext =& $wbSubCs[$i];
			$full = $this->append($baseContext, $subContext);
			if (empty($this->mRemappings[$full])) {
				$mapping = $subContext;
			} else	$mapping = $this->mRemappings[$full];
			$wbRemapToSubC[$mapping] = $subContext;
			$subContext = $mapping;
		}
		return $wbRemapToSubC;
	}

	function sortImportBranch(&$vb) {
		$ctxa = $tmp = array();
		for ($i = 0; $i < count($vb); $i++) {
			$vn = $vb[$i];
			if ($vn['type'] != 'block') {
				$ctxa[$i] = $vn['dispname'];
			}
		}
		natsort($ctxa);
		$tmp = array();
		foreach ($ctxa as $idx => $ctx) {
			$tmp[] = $vb[$idx];
		}
		# "$vb =& $tmp; and make it visible to the caller too"
		array_splice($vb, 0, count($vb), $tmp);
	}

	function processBlock($block, $wb, &$wbRemappedSCs, &$wbRemapToSubC,
	  $baseContext, $visContext, $paramsToStore, $propagated, $pruneLeafs,
	  $noRecurses) {
		$first   = isset($block['first'  ]) ? $block['first'  ] : '';
		$last    = isset($block['last'   ]) ? $block['last'   ] : '';
		$exclude = isset($block['exclude']) ? $block['exclude'] : '';
		$nodes = array();
		if (empty($first)) {
			$pos = $wbRemappedSCs ? 0 : false;
		} else	$pos = findfirstgte_natsort($first,$wbRemappedSCs);
		if ($pos !== false) {
			# assumes no duplicates in $wbRemappedSCs
			if ($wbRemappedSCs[$pos] == $first &&
			  ($exclude == 'first' || $exclude == 'both')) $pos++;
			while ($pos<count($wbRemappedSCs) && (!$last ||
			  (compare_natsort($wbRemappedSCs[$pos], $last) < 0 ||
			   $last == $wbRemappedSCs[$pos] &&
			   $exclude != 'last' && $exclude != 'both'))) {
				$mapping = $wbRemappedSCs[$pos];
				$ctx = $wbRemapToSubC[$mapping];
				$newNode = $this->generateImportedNode(
				  $paramsToStore, $baseContext, $ctx, $mapping,
				  $wb);
				$newVis = $this->append($visContext,
				  $newNode['dispname']);
				$noRecurse = in_array($newNode['title'],
				  $noRecurses);
				# avoid unnecessary + inappropriate calls
				if (count($wb[$ctx]['nodes'])>0 && !$noRecurse){
					$this->mergeImportBranch_r(
					  $newNode['nodes'], $wb[$ctx]['nodes'],
					  $newNode['title'], $newVis,
					  $paramsToStore, $propagated);
				}
				if (!$pruneLeafs || count($newNode['nodes'])>0){
					$this->setNodeId($newNode, $newVis);
					$nodes[] = $newNode;
				}
				unset($wbRemapToSubC[$ctx]);
				array_splice($wbRemappedSCs, $pos, 1);
			}
		}
		return $nodes;
	}

	/**
	 * Doesn't set nodeid: call setNodeId() afterwards.
	 */
	function generateImportedNode($paramsToStore, $baseContext,
	  $subContext, $mapping, $wb, $type = 'importednode') {
		$newNode = $paramsToStore;
		$newNode['type'] = $type;
		$newNode['title'] = $this->append($baseContext, $subContext);
		$newNode['dispname'] = $mapping;
		$aid = isset($wb[$subContext]['aid'])?$wb[$subContext]['aid']:0;
		if ($aid) $newNode['aid'] = $aid;
		$newNode['nodes'] = array();
		return $newNode;
	}

	/**
	 * Prunes nodes (+subtrees) whose conditions aren't met for the current
	 * state and makes necessary replacements of e.g. {USERNAME},
	 * {ARTICLE}, clearing each node member that needs rebuilding.  Also
	 * adjusts each node's expanded/collapsed/leaf state according to the
	 * current request.
	 */
	function updateTreeviewState_r(&$vn) {
		$effectiveAction = $this->mState->getAction();

		$ret = $i = 0;
		while ($i < count($vn['nodes'])) {
			$node =& $vn['nodes'][$i];
			$this->doNodeConditionReplacements($node, false,
			  TV_CONDREPL_NODE);
			if (empty($node['conditions'])) {
				$node['conditions'] = array();
			}
			if (!$this->mConditions->testConditions(
			  $node['conditions'], $node)) {
				array_splice($vn['nodes'], $i, 1);
				continue;
			} else if ($node['type'] != 'importednode' &&
			  count($node['nodes']) <= 0) {
				$j = $i + 1;
				while ($j < count($vn['nodes'])) {
					if(empty($vn['nodes'][$j]['elseflag'])){
						break;
					} else if (count($vn['nodes'][$j]
					  ['nodes']) > 0) {
						$vn['nodes'][$i]['nodes'] =&
						  $vn['nodes'][$j]['nodes'];
						array_splice($vn['nodes'],$j,1);
						break;
					} else {
						array_splice($vn['nodes'],$j,1);
					}
				}
			}

			$newstate = 0;
			$this->doNodeReplacements($node, TV_CONDREPL_PARAMS,
			  false);

			if (empty($node['url']) && empty($node['ext']) &&
			  $node['title']) {
				$t = Title::newFromText($node['title']);
				if (empty($node['urlextra'])) {
					$node['urlextra'] = '';
				}
				$node['url'] = $t ?
				  $t->getLocalURL($node['urlextra']) : '';
				if ($t) $node['title_obj'] = $t;
			}

			if (count($node['nodes']) > 0) {
				$res = $this->updateTreeviewState_r($node);
				if ($res & TV_LEAF) $newstate |= TV_LEAF;
				if ($res & TV_ANCESTOROFSELECTED) {
					$newstate |= TV_ANCESTOROFSELECTED;
					$ret |= TV_ANCESTOROFSELECTED;
				}
			} else	$newstate |= TV_LEAF;

			/* Multiple nodes might be selected e.g. if a title is
			 * imported at separate locations.
			 */
			if ($this->isNodeSelected($node)) {
				$newstate |= TV_SELECTED;
				$ret |= TV_ANCESTOROFSELECTED;
			}

			/* Prevent unexpected collapses. */
			if ($this->mState->isCollapsed($node['id']) &&
			    empty($node['expanded']) &&
			    (!($newstate & TV_ANCESTOROFSELECTED)
			     ||
			     $this->mState->shouldForciblyExpandSelected()
		            )
			   ) $this->mState->removeNodeState($node['id']);

			if ($newstate & TV_LEAF) { /* skip */
			} else if ((!empty($node['expanded']) &&
			    !$this->mState->isCollapsed($node['id']))
			    ||
			    ($this->mState->isExpanded($node['id']))
			    ||
			    (($newstate & TV_ANCESTOROFSELECTED)
			     &&
			     (!$this->mState->isCollapsed($node['id'])
			      ||
			      $this->mState->shouldForciblyExpandSelected()
			    )))	$newstate |= TV_EXPANDED;
			else	$newstate |= TV_COLLAPSED;

			/* Flush the appropriate bits of the cache. */
			if ($node['state'] != $newstate) {
				$node['iconlink'] = null;
				$node['li_start'] = null;
			}

			$node['state'] = $newstate;
			$i++;
		}
		if (count($vn['nodes']) <= 0) $ret |= TV_LEAF;
		return  $ret;
	}

	function isNodeSelected($node) {
		global $wgTV_RedirectedFrom;

		/* Default logic. */
		$c = $this->mConditions;
		$req = $this->mState->getWebRequest();
		$id = $node['id'];
		$title = $node['title'];
		$ret = $c->testCondition(CD_IFNODEIDMATCHES,false,$id)||
		  $req->getVal('nodeid') == '' &&
		  ($c->testCondition(CD_IFTITLEMATCHES, false, $title)||
		   $c->testCondition(CD_IFTITLEMATCHES_POSTREDIR, false, $title)
		  )&&
		  $c->testCondition(CD_IFURLEXTRAMATCHES, false, $node);
		if (!$ret && isset($node['selected'])) {
			$ret = $this->mConditions->testConditions(
			  $node['selected'], $node);
		}

		return $ret;
	}

	/** Extracts from $this->mParameterOptions the list of parameters that
	  * are conditionable, limited to those parameters that are valid for
	  * $nodeType, and returns that list, caching it within the function
	  * for later calls (by implication $this->mParameterOptions and
	  * $this->mParseRules must be 'constant' data).
	  */
	function getConditionableParams($nodeType) {
		static $conditionables = array();
		if (!isset($conditionables[$nodeType])) {
			if (empty($this->mParseRules[$nodeType]['parameters'])){
				$this->mParseRules[$nodeType]['parameters'] =
				  array();
			}
			$paramKeys=$this->mParseRules[$nodeType]['parameters'];
			$po = $this->mParameterOptions;
			$ret = array();
			foreach ($paramKeys as $paramKey) {
				if (!empty($po[$paramKey]['conditionable'])) {
					$ret[] = $paramKey;
				}
			}
			$conditionables[$nodeType] = $ret;
		} else	$ret = $conditionables[$nodeType];
		return $ret;
	}

	/** Replaces variables in the parameters, if any, of conditions, if any.
	  * @param Array   $node  A node from the tree $this->mHierarchy.
	  * @param integer $which The areas in which to replace.  One of:
	  *   TV_CONDREPL_NODE  : replace only in conditions attached to the
	  *                       node itself.
	  *   TV_CONDREPL_PARAMS: replace only in conditions attached to node
	  *                       parameters e.g. "selected".
	  *   TV_CONDREPL_BOTH  : replace both in conditions attached to the
	  *                       node itself and attached to node parameters.
	  */
	function doNodeConditionReplacements(&$node, $which = TV_CONDREPL_NODE,
	  $firstTime = false) {
		$ret = false;
		if (($which & TV_CONDREPL_NODE) &&
		  !empty($this->mParseRules[$node['type']]['conditionable'])) {
			if ($this->doConditionsReplacements(
			  $node['conditions'], $firstTime)) $ret = true;
		}
		if ($which & TV_CONDREPL_PARAMS) {
			$conditionables = $this->getConditionableParams(
			  $node['type']);
			foreach ($conditionables as $k) {
				if ($this->doConditionsReplacements($node[$k],
				  $firstTime)) $ret = true;
			}
		}
		return $ret;
	}

	function doConditionsReplacements(&$conditions, $firstTime = false) {
		$ret = false;
		for ($i = 0; $i < count((array)$conditions); $i++) {
			if ($this->doConditionReplacements($conditions[$i],
			  $firstTime)) $ret = true;
		}
		return $ret;
	}

	function doConditionReplacements(&$condition, $firstTime = false) {
		/* $condition[3] is set to boolean true in parseConditions() if
		 * the condition is replaceable; it might later have been set
		 * to the original pre-replacement string by doReplacements()
		 * if replacements were made. */
		return $condition[3] ? $this->doReplacements($condition,2,3,4,
		  $firstTime) : false;
	}

	/** Replaces variables on a member of an array/map, using supporting
	  * members to record which replacements were made and the
	  * pre-replacement text.  Returns true if replacements were made,
	  * false otherwise.
	  * @param Array $array The array containing the replaceable member.
	  * @param Mixed $kText The key into $array at which the replaced text
	  *   should be stored; if $firstTime is true then this key's value
	  *   should be the original, pre-replacement text.
	  * @param Mixed $kPreRepl The key into $array at which the
	  *   pre-replacement text is stored.  If $firstTime is true then this
	  *   key's value is ignored and it is replaced with $array[$kText] only
	  *   if replacements are made.  Otherwise if no replacements have
	  *   previously been made then this key's value is also ignored.
	  * @param Mixed $kReplMap The key into $array at which the map of
	  *   previously replaced variables is stored, and which will be
	  *   overwritten if replacements are made.  If $firstTime is true
	  *   then this key's existing value is irrelevant.
	  * @return boolean
	  */
	function doReplacements(&$array, $kText, $kPreRepl, $kReplMap,
	  $firstTime = false) {
		$nFullReplMap = $this->mState->getReplacements();
		$oReplMap = isset($array[$kReplMap])?$array[$kReplMap]:array();
		if ($firstTime) {
			if (empty($array[$kText])) $ret = false;
			else {
				$org = $array[$kText];
				list($array[$kText],$repls)=$this->rvars($org);
				$ret = (count($repls) > 0);
				if ($ret) {
					$array[$kPreRepl] = $org;
					$array[$kReplMap] = $repls;
				}
			}
		} else if ($oReplMap &&
		  $oReplMap != array_intersect_key($nFullReplMap, $oReplMap)) {
			list($array[$kText], $array[$kReplMap]) =
			  $this->rvars($array[$kPreRepl]);
			$ret = true;
		} else	$ret = false;
		return $ret;
	}

	function doNodeReplacements(&$node, $whichConditions = TV_CONDREPL_NONE,
	  $firstTime = false) {
		$ret = false;
		if ($whichConditions != TV_CONDREPL_NONE &&
		  $this->doNodeConditionReplacements($node, $whichConditions,
		  $firstTime)) $ret = true;
		$replaceables=$this->mParseRules[$node['type']]['replaceables'];
		foreach ($replaceables as $k) {
			/* Special-case */
			if ($k == 'url' && empty($node['ext'])) continue;
			$res = $this->doReplacements($node, $k,
			   $k.'_prereplace', $k.'_replacements', $firstTime);
			if ($res) $ret = true;
			if ($res && !$firstTime) {
				/* Mark cache empty for rebuilding, assuming
				 * that icons aren't affected by replacements.*/
				$node['link'] = $node['linkdata'] = null;
				if (empty($node['ext'])) {
					$node['url']= $node['title_obj']= null;
					if ($k == 'url' || $k == 'title') {
						$node['aid'] = -1;
					}
				}
			}
		}
		return $ret;
	}

	function buildHierarchyHtml_r(&$vn, $alwaysRecurse = false,
	  $skipStateRebuild = false, $indent = "\t", $ul = false,
	  $ancestorExpands = array()) {
		if (isset($vn['state']) & TV_ANCESTOROFSELECTED) {
			$ancestorExpands[$vn['id']] = TV_EXPANDED;
		}
		$br =& $vn['nodes'];
		# Keep a local cache of article ids in this branch to avoid
		# excessive, expensive calls to getArticleID()
		$aids = array();
		$html = '';
		$viewedTitleEsc = $this->mState->escapeLocalURL();
		for ($k = 0; $k < count($br); $k++) {
			$node =& $br[$k];
			if ($ul && !$html) $html .= "\n".$indent.'<ul>'."\n";
			if (empty($node['id'])) {
				wfDebug(__METHOD__.": node without an id: ".
				  "{$node['dispname']} ({$node['title']})\n");
				$this->setNodeId($node,
				  'NODE:ID:SHOULD:ALREADY:BE:SET');
			}

			if (!empty($node['expanded'])) {
				$this->mExpandeds[] = $node['id'];
			}

			# Check this here rather than in updateTreeviewState_r()
			# because that function modifies the state string.
			$stateStr = $this->mState->getStateStr();
			$rebuildLink = empty($node['link']) ||
			 (!$skipStateRebuild && $stateStr!=$node['tvstatestr']);
			if ($rebuildLink) {
				$this->realiseNodeLink($node, $stateStr, $aids);
			}
			$node['tvstatestr'] = $stateStr;

			if (count($node['nodes']) > 0 &&
			  (($node['state'] & TV_EXPANDED) || $alwaysRecurse)) {
				$childhtml = $this->buildHierarchyHtml_r($node,
				  $alwaysRecurse, $skipStateRebuild,
				  "\t$indent", true, $ancestorExpands);
			} else	$childhtml = '';

			if (isset($node['iconpagebase']) &&
			  $node['iconpagebase'] != $viewedTitleEsc) {
				$node['iconlink'] = '';
			}
			if (empty($node['iconlink'])||empty($node['iconurl']) ||
			  !empty($node['icons']) && empty($node['icondata']) ||
			  $rebuildLink ||
			  $node['ancestorexpands'] != $ancestorExpands) {
				# XREF-1: The next line represents logic shared
				# with client-side code; for details refer to
				# skins/treeview/README.treeview_javascript
				# under the XREF-1 heading.
				$stateStrForIcon = $this->mState->
				  getStateStr($ancestorExpands);
				$this->realiseNodeIcon($node, $stateStrForIcon);
				$node['iconpagebase'] = $viewedTitleEsc;
				$node['ancestorexpands'] = $ancestorExpands;
			}

			if (empty($node['li_start'])) {
				if ($node['state'] & TV_SELECTED) {
					$class = ' class="selected"';
				} elseif ($node['state']&TV_ANCESTOROFSELECTED){
					$class = ' class="ancestorofselected"';
				} else	$class = '';
				$node['li_start'] = $indent.'<li id="'.
				  $this->idForLi($node).'"'.$class.'>';
			}
			$html .= $node['li_start'].$node['iconlink'].
			  $node['link'].$node['linkdata'];
			/* Avoid E_NOTICE warnings; unset if no icons. */
			$html .= isset($node['icondata'])?$node['icondata']:'';
			if ($node['state'] & TV_EXPANDED) $html .= $childhtml;
			$html .= $indent.'</li>'."\n";
		}
		if ($ul && $html) $html .= $indent.'</ul>'."\n";
		return $html;
	}

	function idForLi($node, $reverse = false) {
		if ($node['state'] & TV_LEAF) $c = 'L';
		else if ($node['state'] & TV_EXPANDED) $c = $reverse?'c':'e';
		else	$c = $reverse?'e':'c';
		return 'tv.'.$c.'_'.$node['id'];
	}

	function realiseNodeLink(&$node, $stateStr, &$aids) {
		$indent = ''; /** @todo set value appropriately & check usage*/
		$linker = /* class_exists('DummyLinker') ? new DummyLinker() : */new Linker();

		# $node['title_obj'] is pruned prior to caching but might have
		# been regenerated by updateTreeviewState_r() for later fetches.
		$t = isset($node['title_obj']) ? $node['title_obj'] : null;
		if (!$t && !empty($node['title'])) {
			$t = Title::newFromText($node['title']);
			$node['url'] = $t ? $t->getLocalURL() : '';
		}
		if (!isset($node['aid']) || $node['aid'] == -1) {
			$node['aid'] = isset($node['title']) &&
			  isset($aids[$node['title']]) ?
			  $aids[$node['title']] : ($t ? $t->getArticleID() : 0);
		}
		if (isset($node['title']) && !isset($aids[$node['title']])) {
			$aids[$node['title']] = $node['aid'];
		}
		$node['linkdata'] = '';
		$tooltip = !empty($node['tooltip']) ?
		  ' title="'.htmlspecialchars($node['tooltip']).'" ' : '';
		if (empty($node['dispname'])) $node['dispname'] = '';
		if ($t) {
			$new = (empty($node['aid']) &&
			  empty($node['knownlink']) &&
			  $t->getNamespace() != NS_SPECIAL);
			$extraArr = !empty($node['appendnodeid']) ?
			  array('nodeid' => $node['id']) : array();
			$uExtra = isset($node['urlextra'])?$node['urlextra']:'';
			$extraArr = array_merge(TvQueryString::queryStrToArr(
			  $uExtra), $extraArr);
			if ($new) $extraArr = TvQueryString::
			  convertViewAction($extraArr, 'edit');
			if ($stateStr) {
				$extraArr['tvstate'] = $stateStr;
				/* Avoid file-caching */
				$extraArr = TvQueryString::
				  convertViewAction($extraArr);
			}
			$customAttribs = array('rel' => 'nofollow');
			if ($new) $customAttribs['class'] = 'new';
			if (!empty($node['tooltip'])) {
				$customAttribs['title'] = $node['tooltip'];
			}
			# Avoid makeLink() - it's a lot slower because it has
			# to call getArticleID(); id is already known at this
			# point for 'importednode's because of the batch DB
			# query and for other types it's already been
			# determined by a call to getArticleID() in
			# updateTreeviewState_r() or above in this function.
			$node['link'] = $linker->link($t,
			    htmlspecialchars($node['dispname']), $customAttribs,
			    $extraArr, 'noclasses');
			$node['linkdata'] .= $indent.'<input type="hidden" '.
			  "id=\"dbttl.{$node['id']}\" value=\"".
			  htmlspecialchars(urlencode($t->getPrefixedDBkey())).
			  '" />'."\n".
			  $indent.'<input type="hidden" id="baseurl.'.
			    "{$node['id']}\" value=\"".
			    htmlspecialchars($node['url'])."\" />\n";
		} else if (!empty($node['ext'])) {
			$node['link'] = "<a rel=\"nofollow\" href=\"{$node['url']}\"$tooltip>".
			  htmlspecialchars($node['dispname']).'</a>';
		} else	$node['link'] = htmlspecialchars($node['dispname']);
		$node['link'] = '<span class="text">'.$node['link'].'</span>'.
		  "\n";
	}

	function realiseNodeIcon(&$node, $stateStr) {
		global $wgTreeviewIconWidth, $wgTreeviewIconHeight;

		$indent = ''; /** @todo set value appropriately & check usage*/

		# Using the pre-redirect title ensures that the url doesn't
		# change
		$tViewed = $this->mState->getTitle_Predir();

		if ($node['state'] & TV_LEAF) $idx = 0;
		else $idx = ($node['state'] & TV_EXPANDED) ? 2 : 1;
		if (!isset($node['iconidx']) || $node['iconidx'] !== $idx ||
		  empty($node['iconurl'])) {
			$node['iconidx'] = $idx;
			$node['iconurl'] = '';
			if (!empty($node['icons'])) {
				$node['icondata'] = '';
				for ($i = 0; $i < ICONS_PER_NODE; $i++) {
					if (empty($node['icons'][$i])) continue;
					$img = MediaWikiServices::getInstance()->getRepoGroup()->findFile($node['icons'][$i]);
					if ($img && method_exists($img, 'getUrl')) {
						$customimgurl = $img->getURL();
						if ($i == $idx) {
							$node['iconurl'] =
							  $customimgurl;
						}
						$node['icondata'] .= $indent.
						  '<input type="hidden" id="'.
						  "icon.$i.{$node['id']}\"".
						  ' value="'.$customimgurl.
						  '" />'."\n";
					}
				}
			}
			if (empty($node['iconurl'])) {
				$node['iconurl'] = $this->mDefaultIcons[$idx];
			}
		}

		# Don't append a collapse or expand directive when that state is
		# the default - this assists client-side caching by avoiding
		# duplicate urls for the same page and treeview state
		if (($node['state'] & TV_EXPANDED) &&
		    (!empty($node['expanded']) ||
		    ($node['state'] & TV_ANCESTOROFSELECTED))) {
			$newexpstate = 'c';
		} else if (($node['state'] & TV_COLLAPSED) &&
		    empty($node['expanded']) &&
		    !($node['state'] & TV_ANCESTOROFSELECTED)) {
			$newexpstate = 'e';
		} else	$newexpstate = '';
		$iconStateStr = '';
		if ($newexpstate || $stateStr) {
			if ($newexpstate) {
				$iconStateStr = $newexpstate.'_'.$node['id'];
			}
			if ($stateStr) {
				if ($qryappendstr = preg_replace(
				  "/(^|,)[ec]_{$node['id']}(,|$)/", '$1',
				  $stateStr)) {
					if ($iconStateStr) $iconStateStr .= ',';
					$iconStateStr .= $qryappendstr;
				}
			}
		}

		if ($node['state'] & TV_LEAF) {
			$hint = wfMessage('tv_tt_leafnode')->escaped();
			$alt = wfMessage('tv_txtleafimg')->escaped();
		} else if ($node['state'] & TV_EXPANDED) {
			$hint = wfMessage('tv_tt_collapsenode')->escaped();
			$alt = wfMessage('tv_txtcollapseimg')->escaped();
		} else {
			$hint = wfMessage('tv_tt_expandnode')->escaped();
			$alt = wfMessage('tv_txtexpandimg')->escaped();
		}
		$node['iconlink'] = '';
		if (!($node['state'] & TV_LEAF)) {
			$qrystr = $this->mState->getQueryStr();
			$act = $this->mState->getAction();
			$qa = TvQueryString::queryStrToArr($qrystr);
			$qa = TvQueryString::convertViewAction($qa, $act);
			# Don't append samepage=1 when the state is empty, to
			# avoid dup'd urls as above
			if ($iconStateStr) {
				$qa['samepage'] = '1';
				$qa['tvstate'] = $iconStateStr;
				$qa = TvQueryString::convertViewAction($qa);
			}
			$qrystr = TvQueryString::arrToQueryStr($qa);
			$node['iconlink'] = '<a class="icon" rel="nofollow" '.
			  'href="'./*htmlspecialchars($tViewed->
			  getLocalURL($qrystr)).*/'#'./*$this->
			  idForLi($node, true).*/"\">";
		}
		$node['iconlink'] .= "<img alt=\"$alt\" title=\"$hint\" ".
		  "width=\"$wgTreeviewIconWidth\" ".
		  "height=\"$wgTreeviewIconHeight\"";
		if ($node['state'] & TV_LEAF) {
			$node['iconlink'] .= ' class="icon"';
		}
		$node['iconlink'] .= ' src="'.$node['iconurl'].'" />';
		if (!($node['state'] & TV_LEAF)) {
			$node['iconlink'] .= '</a>';
		}
		$node['iconlink'] .= ' ';
	}

	/**#@- */ # end private block
}

function wfLogTreeview($logMsg) {
	global $wgTreeviewLogfile;
	if ($wgTreeviewLogfile) {
		if ($handle = fopen($wgTreeviewLogfile, 'a')) {
			fwrite($handle, strftime('%c').': '.$logMsg);
			fclose($handle);
		}
	}
}

?>
