<?php
/** QueryString.php
  *
  * Part of the Hierarchy extension for MediaWiki; supporting the Treeview skin.
  *
  * Licenced under the General Public Licence version 2, without warranty -
  * refer to the LICENCE file which in the unlikely event of differences
  * supercedes the url below.
  *
  * @licence GPL2 http://www.gnu.org/copyleft/gpl.html
  * @author Laird Shaw <lairdshaw77@gmail.com>
  */

if (!defined('MEDIAWIKI')) die();

/** For xsplit2() */
require_once('UtilityFunctions.php');

/** This class provides several functions related to url query strings and the
  * normalisation of those strings for Hierarchy/Treeview purposes.  All of the
  * functions are statically callable i.e. the class is simply a namespace
  * container.
  *
  * @package MediaWiki
  * @subpackage Extensions
  */
class TvQueryString {
	/** Converts the url query string $queryStr into an array and returns it
	  * after excluding treeview-specific query parameters (unless $exclude
	  * is false) and url-decoding each parameter key and value.
	  * Does not ensure that the 'action' parameter has an entry, nor does
	  * it convert synonyms of the 'view' action to 'view': both of those
	  * tasks are performed by convertViewAction().
	  * @param string $queryStr The part of a url that follows the ? symbol.
	  * @return Array Query parameter values, keyed by parameter name.
	  * @static.
	  */
	static function queryStrToArr($queryStr, $exclude = true) {
		/**
		 * The set of query parameters that can be ignored for
		 * comparison purposes, assuming that title is compared
		 * separately.
		 * @note Synchronise changes with the list in the description
		 * of ifurlextramatches under "Supported conditions" in
		 * help-articles/Help.Treeview_skin.Syntax.wiki.
		 */
		static $excludedParams = array('title', 'samepage', 'tvstate',
		  'skiplinkstaterebuild');
		$ret = array();
		$a = $queryStr ? explode('&', $queryStr) : array();
		foreach ($a as $p) {
			list($k, $v) = xsplit2('=', $p);
			$k_dec = urldecode($k);
			if (!$exclude || !in_array($k_dec, $excludedParams)) {
				$ret[$k_dec] = urldecode($v);
			}
		}
		return $ret;
	}

	/** Reconstructs a query string from a parameter array.
	  * Url-encodes each parameter key and value.
	  * @param Array Query parameter values, keyed by parameter name.
	  * @return string The part of a url after the ? symbol.
	  * @static.
	  */
	static function arrToQueryStr($params) {
		$a = array();
		foreach ($params as $k => $v) {
			$a[] = urlencode($k).'='.urlencode($v);
		}
		return implode('&', $a);
	}

	/** The set of equivalencies from a Treeview/Hierarchy perspective for
	  * the 'view' action.  Would be a static member but that's not
	  * supported for earlier PHP versions.
	  * @static.
	  */
	static function getViewSynonyms() {
		return array('', 'view', 'view_tv');
	}

	/** Sets the value of the 'action' parameter to $newActionValue, but
	  * only if it was originally unset/empty or otherwise synonymous for
	  * Treeview/Hierarchy purposes with 'view'.
	  * @param Array $params Query parameter values, keyed by parameter name
	  * @param string $newActionValue
	  * @return Array The (un)modified input array.
	  * @static.
	  */
	static function convertViewAction($params, $newActionValue = 'view_tv'/*default
	 so that this string appears only in this file: simpler maintenance.*/){
		$viewSynonyms = TvQueryString::getViewSynonyms();
		if (!isset($params['action'])) $params['action'] = '';
		if (in_array($params['action'], $viewSynonyms)) {
			$params['action'] = $newActionValue;
		}
		return $params;
	}
}

?>
