<?php
/**
 * State.php
 *
 * Part of the Hierarchy extension for MediaWiki; supporting the Treeview skin.
 *
 * Licenced under the General Public Licence 2, without warranty.
 *
 * @licence GPL2 http://www.gnu.org/copyleft/gpl.html
 * @author Laird Shaw <lairdshaw77@gmail.com>
 */

if (!defined('MEDIAWIKI')) die();

use MediaWiki\MediaWikiServices;
use MediaWiki\Request;
use MediaWiki\Request\FauxRequest;
use MediaWiki\Title\Title;

global $IP;
/** For FauxRequest so that FauxRequestEx can extend it. */
require_once("$IP/includes/Request/WebRequest.php");
/** For TvQueryString::queryStrToArr(). */
require_once('QueryString.php');
/** For Revision::newFromTitle(). */
# seems to cause problems with MW 1.7.x and to be redundant anyway
//require_once("$IP/includes/Revision.php");

class FauxRequestEx extends FauxRequest {
	var $mFullRequestURL;

	function __construct($data, $fullRequestURL, $wasPosted = false) {
		if (method_exists('FauxRequest', 'FauxRequest'))
			parent::FauxRequest($data, $wasPosted);
		else
			parent::__construct($data, $wasPosted);
		$this->mFullRequestURL = $fullRequestURL;
	}

	function getRequestURL() {
		return $this->mRequestURL;
	}
}

# Keep this constant in sync with its object-key counterpart in treeview.js
define('TVSYNCSERVER'         ,  'server');

# These are used in bitmasks - even though some of them are mutually exclusive
# - so stick to powers of two.
define('TV_LEAF'              ,  1);
define('TV_SELECTED'          ,  2);
define('TV_ANCESTOROFSELECTED',  4);
define('TV_EXPANDED'          ,  8);
define('TV_COLLAPSED'         , 16);

/** This class encapsulates from the client browser's point of view the state
  * of the treeview/hierarchy used by the Treeview skin.
  *
  * It allows effective values to be specified for global objects such as
  * $wgTitle / $wgArticle->getTitle() and $wgRequest, as well as for the
  * expanded state of the treeview to be specified.  Alternate states are
  * necessary for two reasons:
  * a) because load-on-demands through the HierarchyPage class's
  *    Special:Hierarchy are on behalf of a different page and it is necessary
  *    to have access to the state of the represented page as it was originally
  *    browsed to by the user agent, rather than the state at the point at
  *    which the javascript request to Special:Hierarchy occurs.
  * b) because prior to caching the treeview, its html and node attributes are
  *    generated based on a likely default state rather than the current state -
  *    this avoids too much need for change when rebuilding after cache
  *    retrieval.
  *
  * @note The mQueryStr and mWebRequest members are not validated for
  * consistency with each other - it is up to clients of this class to
  * read/write them sensibly.
  *
  * @package MediaWiki
  * @subpackage Extensions
  */
class TVstate {
	/** The effective title, post-redirect; if ===false then not a redirect
	  * else if null then $mTitlePreRedirect hasn't been checked for a
	  * redirect or followed yet.
	  * @var Title */
	var $mTitlePostRedirect = null;
	/** The effective title prior to any redirects
	  * @var Title */
	var $mTitlePreRedirect;
	var $mUser;
	var $mWebRequest;
	var $mTVstates;
	var $mQueryStr;

	/** Caches for getQueryArr(). */
	var $mQueryArr;
	var $mQueryArr_norm;

	var $mAction;
	var $mReplacements;
	/** A cache used only by escapeLocalURL(). */
	var $mEscLocalURL;

	var $mForceExpandSelected;
	var $mOriginalStateStr;

	/**#@+ @access public */
	/**
	 * The constructor for TVstate objects.
	 *
	 * Set a parameter to (int)-1 to indicate that a "default" state should
	 * be used - this is based on an anonymous view of the 'Main Page'
	 * article with nothing expanded.  Leave a value unset (or avoid passing
	 * $paramsArr at all) to indicate that the applicable global variable
	 * or other sensible value for the "current" state should be used.
	 *
	 * The parameter keys and expected values are:
	 * 'tEffective' => Title object qualified by 'redirectInfo', if
	 *                 null/unset then it defaults to
	 *                   $wgArticle->getTitle() if set, else $wgTitle;
	 *                 if -1 then it defaults to the main page.
	 * 'redirectInfo' => Mixed; can be one of:
	 *   null/unset  : tEffective is pre-redirect; its redirect-status is
	 *                 unchecked (this TVstate object will check/follow it
	 *                 if and when first required, caching the result).
	 *   Title object: tEffective is post-redirect; this is the redirected-
	 *                 from title
	 *   false       : not a redirect; equivalent to null if tEffective is
	 *                 also null
	 * 'effectiveUser'       => User object, defaults to $wgUser
	 * 'effectiveWebRequest' => A WebRequest or FauxRequestEx object.
	 * 'TVexpandState'       => An array of bitmasks - the most relevant
	 *                          defines being TV_EXPANDED and TV_COLLAPSED -
	 *                          indexed by nodeid.
	 * 'autoExpandSelected'  => Boolean.  Whether to consider the 'samepage'
	 *                          url parameter to be unset.
	 * 'effectiveQueryStr'   => String.  The part of the url following the ?
	 * 'effectiveAction'     => String.  The action, defaulting to 'view'.
	 * @param Array $params
	 */
	function __construct($params = array()) {
		global $wgArticle, $wgRequest, $action, $wgTitle;
		static $nulls = array('tEffective' => null,
		  'redirectInfo' => null, 'effectiveUser'=> null,
		  'effectiveWebRequest' => null, 'TVexpandState' => null,
		  'autoExpandSelected' => null, 'effectiveQueryStr' => null,
		  'effectiveAction' => null);
		/* Avoid E_NOTICE errors due to unset keys. */
		$pa = array_merge($nulls, $params);;
		if ($pa['tEffective'] === -1) {
			$tEffective = Title::newFromText(wfMessage('mainpage')->plain());
		} else $tEffective = !is_null($pa['tEffective']) ?
		  $pa['tEffective'] : ($wgArticle ? $wgArticle->getTitle() : $wgTitle);
		if ($pa['redirectInfo'] === false) {
			$this->mTitlePostRedirect = false;
			$this->mTitlePreRedirect = $tEffective;
		} else if (!is_null($pa['redirectInfo'])) {
			$this->mTitlePreRedirect = $pa['redirectInfo'];
			if (!is_null($pa['tEffective'])) {
				# N.B. In later versions of MW core (at least
				# 1.23, probably earlier), we set the wrong
				# value here, because $wgTitle, to which
				# $pa['tEffective'] has been set, is
				# pre-redirect title. We fix this at the bottom
				# of the method.
				$this->mTitlePostRedirect = $pa['tEffective'];
			}
		} else	$this->mTitlePreRedirect = $tEffective;

		$this->mUser = !is_null($pa['effectiveUser']) ?
		  $pa['effectiveUser'] : RequestContext::getMain()->getUser();
		if ($this->mUser === -1) {
			/* Anonymous user if not otherwise specified */
			$this->mUser = new User();
			$this->mUser->setId(0);
		}
		if ($pa['effectiveWebRequest'] === -1) {
			$this->mWebRequest = new FauxRequestEx(array(), '');
		} else	$this->mWebRequest =
		  !is_null($pa['effectiveWebRequest']) ?
		  $pa['effectiveWebRequest'] : $wgRequest;
		if ($pa['TVexpandState'] === -1) $this->mTVstates = array();
		else if (!is_null($pa['TVexpandState'])) {
			$this->mTVstates = $pa['TVexpandState'];
		} else $this->loadFromQuery();
		if (!is_null($pa['autoExpandSelected'])) {
			$this->mForceExpandSelected = $pa['autoExpandSelected'];
			if ($this->mForceExpandSelected === -1) {
				$this->mForceExpandSelected = true;
			}
			wfDebug('TVstate:: set "forcible expand for selected" '.
			  'from constructor arguments to "'.
			  $this->mForceExpandSelected.'"'."\n");
		}

		if ($pa['effectiveQueryStr'] === -1) $this->mQueryStr = '';
		else $this->mQueryStr = !is_null($pa['effectiveQueryStr']) ?
		  $pa['effectiveQueryStr'] : $_SERVER['QUERY_STRING'];
		$this->mQueryArr_norm = null;

		if ($pa['effectiveAction'] === -1) $this->mAction = 'view';
		else if (!is_null($pa['effectiveAction'])) {
			$this->mAction = $pa['effectiveAction'];
		} else {
			$action_wr = $this->mWebRequest->getVal('action');
			if ($action_wr) $this->mAction = $action_wr;
			else {
				$qArr = $this->getQueryArr();
				if (isset($qArr['action'])) {
					$this->mAction = $qArr['action'];
				} else if (is_null($pa['effectiveWebRequest'])
				  && is_null($pa['effectiveQueryStr']) &&
				  $action != '') {
					$this->mAction = $action; /*global var*/
				} else	$this->mAction = 'view';
			}
		}

		# The fix referenced above.
		if (is_object($this->mTitlePostRedirect) &&
		    $this->mTitlePostRedirect->isRedirect()) {
			$this->mTitlePostRedirect = null;
			$this->testResolveRedirect();
		}

	}

	/** Returns the effective user object. (should) Never (be) null. */
	function getUser() {
		return $this->mUser;
	}

	/** Returns the effective webrequest object. */
	function getWebRequest() {
		return $this->mWebRequest;
	}

	/** Returns the url query string for the page's effective web request.
	  */
	function getQueryStr() {
		return $this->mQueryStr;
	}

	/** Returns the url query string as an array of urldecoded values keyed
	  * by parameter.  If $norm is true then the action parameter is
	  * normalised: synonyms of 'view', including when there is no
	  * action parameter, are converted to 'view'.
	  * @param boolean $norm
	  * @return Array
	  */
	function getQueryArr($norm = false) {
		if ($norm && $this->mQueryArr_norm === null ||
		    !$norm && $this->mQueryArr === null) {
			$qStr = $this->getQueryStr();
			$this->mQueryArr = TvQueryString::queryStrToArr($qStr);
			if ($norm) $this->mQueryArr_norm = TvQueryString::
			  convertViewAction($this->mQueryArr, 'view');
		}
		return $norm ? $this->mQueryArr_norm : $this->mQueryArr;
	}

	/** Returns the effective action for the effective page's web request.
	  */
	function getAction() {
		return $this->mAction;
	}

	/** Returns the effective url - excluding server component - for the
	  * page based on the effective web request.  Might be empty.
	  */
	function getRequestURL() {
		return $this->mWebRequest->getRequestURL();
	}

	/**
	 * Returns the title object of the effective current page, with
	 * redirect (if any) already followed.  Might be null.
	 */
	function getTitle() {
		return $this->redirected() ? $this->mTitlePostRedirect :
		  $this->mTitlePreRedirect;
	}

	/**
	 * Returns the title object of the effective current page, without
	 * following the redirect if one exists.  Might be null.
	 */
	function getTitle_Predir() {
		return $this->mTitlePreRedirect;
	}

	function redirected() {
		$this->testResolveRedirect();
		return $this->mTitlePostRedirect ? true : false;
	}

	function escapeLocalURL() {
		if (!$this->mEscLocalURL && $this->mTitlePreRedirect) {
			$this->mEscLocalURL = htmlspecialchars($this->
			  mTitlePreRedirect->getLocalURL());
		}
		return $this->mEscLocalURL;
	}

	function getOriginalStateStr() {
		return $this->mOriginalStateStr;
	}

	function getStateStr($overlaystate = array()) {
		$ret = '';
		$state_arr = array_merge($this->mTVstates, $overlaystate);
		/** @todo sort alphabetically; likewise in js */
		foreach ($state_arr as $nodeid => $s) {
			if ($s == 0) continue;
			if ($ret) $ret .= ',';
			$ret .=  (($s & TV_EXPANDED) ? 'e' : 'c')."_$nodeid";
		}
		return $ret;
	}

	function isExpanded($nodeid) {
		if (empty($this->mTVstates[$nodeid])) {
			$this->mTVstates[$nodeid] = 0;
		}
		return $this->mTVstates[$nodeid] & TV_EXPANDED;
	}

	function isCollapsed($nodeid) {
		if (empty($this->mTVstates[$nodeid])) {
			$this->mTVstates[$nodeid] = 0;
		}
		return $this->mTVstates[$nodeid] & TV_COLLAPSED;
	}

	function removeNodeState($nodeid) {
		wfDebug('TVstate::removeNodeState: unsetting $this->mTVstates['.
		  "$nodeid] with value \"{$this->mTVstates[$nodeid]}\"\n");
		unset($this->mTVstates[$nodeid]);
	}

	/** Returns true if all ancestors of the selected node(s) should be
	  * expanded to reveal it/them regardless of the stored collapsed/
	  * expanded state of those ancestor nodes
	  */
	function shouldForciblyExpandSelected() {
		return $this->mForceExpandSelected;
	}

	function getReplacements() {
		if (!$this->mReplacements) {
			$user = $this->getUser();
			$request = $this->getWebRequest();
			$tPage = $this->getTitle();
			$tTalk = $tPage->canHaveTalkPage() ?
			  $tPage->getTalkPage() : null;
			$tArticle = $tPage->getSubjectPage();
			$tPage_pre = $this->getTitle_Predir();
			$tTalk_pre = $tPage_pre->canHaveTalkPage() ?
			  $tPage_pre->getTalkPage() : null;
			$tArticle_pre = $tPage_pre->getSubjectPage();
			$this->mReplacements = array(
				'{{USERNAME}}' => $user->getName(),
				'{{ARTICLEPAGENAME}}'=>$tArticle->
				  getPrefixedText(),
				'{{TALKPAGENAME}}' => $tTalk ?
				  $tTalk->getPrefixedText() : null,
				'{{FULLPAGENAME}}' => $tPage->getPrefixedText(),
				'{{ARTICLEPAGENAME_PREDIR}}' => $tArticle_pre->
				  getPrefixedText(),
				'{{TALKPAGENAME_PREDIR}}' => $tTalk_pre ?
				  $tTalk_pre->getPrefixedText() : null,
				'{{FULLPAGENAME_PREDIR}}' => $tPage_pre->
				  getPrefixedText(),
				'{{REVISIONID}}' => $request ?
				  $request->getVal('oldid') : '',
				'{{USERPAGEUSERNAME}}' => Conditions::
				  isUserPage($tPage) ? Conditions::
				    getFirstSubPageOrContext($tPage) : ''
			);
		}
		return $this->mReplacements;
	}
	/**#@- */ # end public section

	/**#@+ @access private */
	/**
	 * Resolves any redirect if required; stores the result.  If this
	 * becomes too slow within load-on-demands it might be possible to
	 * speed it up by instead querying the links table.
	 */
	function testResolveRedirect() {
		if (!$this->mTitlePostRedirect &&
		  $this->mTitlePostRedirect !== false) {
			$tTarget = false;
			if ($this->mTitlePreRedirect->isRedirect()) {
				if (($wikiPage = MediaWikiServices::getInstance()->getWikiPageFactory()->newFromTitle($this->
				  mTitlePreRedirect))) {
					$tTarget = $wikiPage->
					  getRedirectTarget();
				}
			}
			$this->mTitlePostRedirect = $tTarget ? $tTarget : false;
		}
	}

	function loadFromQuery() {
		global $wgRequest;

		$this->mTVstates = array();
		$ca = explode(',', isset($_COOKIE['tvstate_ui']) ? $_COOKIE['tvstate_ui'] : '');
		if (in_array(TVSYNCSERVER.'_synctype', $ca) &&
		  $this->mUser->isRegistered()) {
			$this->extractTVstate($_COOKIE['tvstate'],
			  $this->mTVstates);
		}
		if (isset($_GET['tvstate'])) {
			$this->extractTVstate($_GET['tvstate'],
			  $this->mTVstates);
		}
		# Always show ancestors of the selected node(s) as expanded if
		# we reached this page from a different one.
		$this->mForceExpandSelected = !$wgRequest->getBool('samepage');
		if ($this->mForceExpandSelected) {
			wfDebug('TVstate::loadFromQuery: set "forcible expand '.
			  'for selected" true'."\n");
		}
		$this->mOriginalStateStr = $this->getStateStr();
		wfDebug('TVstate::loadFromQuery: determined client tv state '.
		  'as: '.var_export($this->mTVstates, true)."\n");
	}

	function extractTVstate($text, &$states) {
		$cnt = 0;
		foreach (explode(',', $text) as $ndstate) {
			if (substr($ndstate, 0, 2) == 'e_') $s = TV_EXPANDED;
			else if (substr($ndstate, 0, 2) == 'c_') {
				 $s = TV_COLLAPSED;
			} else continue; # ignore malformed component
			$states[substr($ndstate, 2)] = $s;
			$cnt++;
		}
		return $cnt;
	}
	/**#@- */ # end private section
}

?>
