<?php

use MediaWiki\MediaWikiServices;
use MediaWiki\Title\Title;

require_once 'Hierarchy.php';

class TreeviewHooks {
	/** Adds tabs for the rebuildtreeview, parsetreeview, treeviewcachestatus
	* actions when the current page is the (pseudo-) Treeview/Hierarchy
	* configuration page.
	*/
	public static function onSkinTemplateNavigation__Universal($sktemplate, &$content_navigation): void {
		global $wgArticle, $wgTitle, $wgRequest;
		$t = ($wgArticle ? $wgArticle->getTitle() : $wgTitle);
		$action = $wgRequest->getText('action');
		$res = Hierarchy::isConfigPage($t);
		if ($res !== false) {
			if (RequestContext::getMain()->getUser()->isAllowed('parsetreeview')) {
				$content_navigation['actions']['parsetreeview'] = array(
					'class' => $action == 'parsetreeview' ?
						'selected' : 'tvaction',
					'text'  => wfMessage('tv_parsetreeview')
						->escaped(),
					'href'  => $t->getLocalUrl(
						'action=parsetreeview')
				);
			}
			if (RequestContext::getMain()->getUser()->isAllowed('rebuildtreeview')) {
				$content_navigation['actions']['rebuildtreeview'] = array(
					'class' => $action == 'rebuildtreeview' ?
						'selected' : 'tvaction',
					'text'  => wfMessage('tv_rebuildtreeview')
						->escaped(),
					'href'  => $t->getLocalUrl(
						'action=rebuildtreeview')
				);
			}
			if (RequestContext::getMain()->getUser()->isAllowed('treeviewcachestatus')) {
				$content_navigation['actions']['treeviewcachestatus'] = array(
					'class' => $action == 'treeviewcachestatus' ?
						'selected' : 'tvaction',
					'text'  => wfMessage('tv_treeviewcachestatus')
						->escaped(),
					'href'  => $t->getLocalUrl(
						'action=treeviewcachestatus')
				);
			}
		}
	}

	public static function onArticleViewRedirect($article) {
		global $wgTV_RedirectedFrom;
		$sk = RequestContext::getMain()->getSkin();
		$wgTV_RedirectedFrom = $article->mRedirectedFrom;

		return (isset($sk->mDisableDefaultRedirectSubHeadings) &&
			$sk->mDisableDefaultRedirectSubHeadings)
			? false
			: true;
	}
	
	public static function onArticleDeleteComplete( $wikiPage, $user, $reason, $id,
		$content, $logEntry, $archivedRevisionCount
	) {
		wfDebug(__CLASS__.'::'.__METHOD__.": Invoking indiscriminate invalidation.\n");
		wfHierarchy_invalidateAll();
	}

	public static function onArticleUndelete() {
		wfDebug(__CLASS__.'::'.__METHOD__.": Invoking indiscriminate invalidation.\n");
		wfHierarchy_invalidateAll();
	}

	public static function onFileUndeleteComplete() {
		wfDebug(__CLASS__.'::'.__METHOD__.": Invoking indiscriminate invalidation.\n");
		wfHierarchy_invalidateAll();
	}

	public static function onPageMoveComplete() {
		wfDebug(__CLASS__.'::'.__METHOD__.": Invoking indiscriminate invalidation.\n");
		wfHierarchy_invalidateAll();
	}

	public static function onPageContentSave($wikiPage) {
		global $wgSpecialHierarchySavedHookData;

		$hookData = array();

		$hookData['old_categories'] = [];
		foreach ($wikiPage->getCategories() as $title) {
			$hookData['old_categories'] = $title->getText();
		}

		$t = $wikiPage->getTitle();
		$id = $t->getArticleID();
		# Trigger on all page creations (this test could be refined) ...
		$invalidate = ($id == 0);
		if ($invalidate) wfDebug(__CLASS__.'::'.__METHOD__.": detected a page creation\n");
		else {
			# ...and on edits to the hierarchy config page...
			if (Hierarchy::isConfigPage($t)) {
				$invalidate = true;
				wfDebug(__CLASS__.'::'.__METHOD__.": detected a save to the config page\n");
			}
		}
		# ... and on edits to any messages upon which that config depends
		if (!$invalidate) {
			foreach (Hierarchy::getMessageDeps() as $msg) {
				# MediaWiki always capitalises the first letter of
				# articles in the message namespace upon save even when
				# $wgCapitalLinks = false
				$tMsg = Title::newFromText(ucfirst($msg), NS_MEDIAWIKI);
				if ($id == $tMsg->getArticleID()) {
					$invalidate = true;
					wfDebug(__CLASS__.'::'.__METHOD__.": detected a saved edit to the ".
					"message '$msg', upon which the config page ".
					"depends\n");
					break;
				}
			}
		}
		if ($invalidate) {
			$hookData['invalidatedall'] = true;
		} else {
			# ... and possibly later if this is a changing redirect
			$tTarget = false;
			if (($wikiPage = MediaWikiServices::getInstance()->getWikiPageFactory()->newFromTitle($t))) {
				$c = $wikiPage->getContent();
				$tTarget = $c->getRedirectTarget();
				if ($tTarget) {
					$hookData['old_target'] = $tTarget;
					$a = wfGetRedirectsTo($tTarget);
					$a[] = $tTarget;
					$hookData['old_aliaslist'] = $a;
				}
			}
//			if ($tTarget) {
//				$hookData = null;
//			}
		}
		$wgSpecialHierarchySavedHookData = $hookData;
	}

	public static function onPageSaveComplete($wikiPage, $user, $summary, $flags, $revisionRecord, $editResult) {
		global $wgSpecialHierarchySavedHookData;
		$hookData = $wgSpecialHierarchySavedHookData;

		$new_categories = [];
		foreach ($wikiPage->getCategories() as $title) {
			 $new_categories[] = $title->getText();
		}
		$old_categories = !empty($hookData['old_categories']) ? $hookData['old_categories'] : [];
		sort($old_categories);
		sort($new_categories);

		$t = $wikiPage->getTitle();
		if (!empty($hookData['invalidatedall']) || $old_categories != $new_categories) {
			wfHierarchy_invalidateAll();
			if (isset($hookData['treeviewtagran'])) {
				# Invalidate the parser cache of the saved article
				# because the treeview tag was built using the
				# invalidated treeview.
				wfDebug(__CLASS__.'::'.__METHOD__.": invalidating the cache of the saved new page due to <treeview> tag use.\n");
				$t->invalidateCache();
			}
			return;
		}
		$tTarget = MediaWiki\MediaWikiServices::getInstance()->getRedirectLookup()->getRedirectTarget($wikiPage);
		$tOldTarget = isset($hookData['old_target']) ? $hookData['old_target'] : null;
		# If this was a redirect that post-save no longer redirects to the same
		# page ...
		if ($tOldTarget && (!$tTarget ||
		                    $tOldTarget->getArticleID() != $tTarget->getArticleID())) {
			# ...then invalidate the article that it previously redirected
			# to and all other articles that redirected to that article.
			$invalArr = $hookData['old_aliaslist'];
			$titleInList = true;
		} else{
			$invalArr = array();
			$titleInList = false;
		}
		# If this is now a redirect that points to a different page than
		# pre-save ...
		if ($tTarget && (!$tOldTarget ||
		$tOldTarget->getArticleID() != $tTarget->getArticleID())) {
			# ...then invalidate the article that it now redirects to and
			# all other articles that redirect to that article.
			$a = wfGetRedirectsTo($tTarget);
			if ($titleInList) $a[] = $tTarget;
			$invalArr = array_merge($invalArr, $a);
		}
		if ($invalArr) {
			$dbgList = '';
			foreach ($invalArr as $t) {
				if ($dbgList) $dbgList .= ', ';
				$dbgList .= $t->getPrefixedText();
			}
			wfDebug(__CLASS__.'::'.__METHOD__.": detected a new or changed redirect - INVALIDATING the cache of these articles: $dbgList\n");
			foreach($invalArr as $title) {
				$title->invalidateCache();
			}
		}
	}

	/**
	 * This is overkill when the image already exists - as long as its url doesn't
	 * change there's no need to rebuild things - but if the upload is for a new
	 * image then this invalidation is required because the url will have been
	 * unknown to the treeview.  An optimisation would be to distinguish between
	 * the two.
	 */
	public static function onUploadComplete($uploadBase) {
		$imageName= $uploadBase->getLocalFile()->getName();
		if (Hierarchy::usesImage($imageName)) {
			wfDebug(__CLASS__.'::'.__METHOD__.": Uploaded image '$imageName' is used by the treeview: invoking indiscriminate invalidation.\n");
			wfHierarchy_invalidateAll();
		}
	}
}
