<?php
/**
 * UtilityFunctions.php
 *
 * Several array and string utility functions that are used by Hierarchy.php
 * and that are not present in PHP core or are missing in earlier PHP versions.
 * Also two MediaWiki-based utility functions used by the Treeview/Hierarchy.
 *
 * Licenced under the General Public Licence 2, without warranty.
 *
 * @licence GPL2 http://www.gnu.org/copyleft/gpl.html
 * @author Laird Shaw <lairdshaw77@gmail.com>
 */

use MediaWiki\MediaWikiServices;
use MediaWiki\Title\Title;

/**
 * Returns all articles that redirect to the article with title $nt.
 * @param Title $nt
 * @return Array of title objects.
 * Title::getLinksTo() would be a drop-in replacement except that it
 * doesn't specify page_is_redirect = 1.
 */
function wfGetRedirectsTo($nt) {
	$redirs = array();
	// $dbw = wfGetDB(DB_MASTER);
	$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
	$ttl = $dbw->strencode($nt->getDBkey());
	$ns = $nt->getNamespace();
	$page = $dbw->tableName('page');
	$pagelinks = $dbw->tableName('pagelinks');
	if ($res = $dbw->query("
		SELECT		page_namespace,
				page_title
		FROM		$pagelinks l
		INNER JOIN	$page p
		ON		l.pl_from = p.page_id
		WHERE		page_title = '$ttl' AND
				pl_from_namespace = $ns AND
				page_is_redirect = 1")) {
		while ($obj = $res->fetchObject()) {
			$redirs[] = Title::makeTitle(
			$obj->page_namespace, $obj->page_title);
		}
	}

	return $redirs;
}

function wfGetPagesInCategories($categories) {
	$ret = array();
	if (count($categories) > 0) {
		// $dbr = wfGetDB(DB_SLAVE, 'category');
		$dbr = MediaWikiServices::getInstance()->getConnectionProvider()->getReplicaDatabase();
		$catTitleWhere = "cat_title in ('";
		foreach ($categories as $category)
			$catTitleWhere .= "', '".$dbr->strencode($category);
		$catTitleWhere .= "')";
		$res = $dbr->select(
		  array('page', 'categorylinks', 'category'),
		  array('page_id', 'page_title', 'page_namespace'),
		  array($catTitleWhere),
		  __FUNCTION__,
		  array(),
		  array('categorylinks' =>
		                       array('INNER JOIN', 'cl_from = page_id'),
		        'category'      => array('LEFT JOIN', 'cl_to = cat_title'))
		);
		while ($x = $res->fetchObject()) 
			$ret[$x->page_id] = 
			  array('page_namespace' => $x->page_namespace,
			        'page_title'     => $x->page_title);
	}

	return $ret;
}

/**
 * Returns all articles that are in any of the same categories as the article
 * with title $nt.  The return is a list of title objects or, if $asArray is
 * true, of array('page_namespace'=>, 'page_title'=>), indexed by page_id.
 * @param Title $nt
 * @return Array.
 */
function wfGetSeeAlsos($nt, $asArray = false) {
	$ret = array();
	// $dbw = wfGetDB(DB_MASTER);
	$dbw = MediaWikiServices::getInstance()->getConnectionProvider()->getPrimaryDatabase();
	$page = $dbw->tableName('page');
	$categorylinks = $dbw->tableName('categorylinks');
	$sql = "
		SELECT DISTINCT	page_namespace,
				page_title,
				page_id
		FROM		$page
		INNER JOIN	$categorylinks
		ON		cl_from = page_id
		WHERE		cl_to in (
			SELECT	c.cl_to
			FROM $page p
			INNER JOIN $categorylinks c
			ON c.cl_from = p.page_id
			WHERE page_namespace=".$nt->getNamespace()."
			AND page_title='".$dbw->strencode($nt->
				getDBkey())."'
		) AND	NOT (page_namespace=".$nt->getNamespace()."
			AND page_title='".$dbw->strencode($nt->
			  getDBkey())."')";
	if ($res = $dbw->query($sql)) {
		while ($obj = $res->fetchObject()) {
			if ($asArray) {
				$ret[$obj->page_id] = array(
					'page_namespace'=>$obj->page_namespace,
					'page_title' => $obj->page_title
				);
			} else {
				$t = Title::makeTitle($obj->page_namespace,
				  $obj->page_title);
				if ($t) $ret[] = $t;
			}
		}
	}
	return $ret;
}

/** Which, if any, of the values in array $starters begins the string $str? */
function whichstarts($str, $starters) {
	foreach ($starters as $starter)
		if (substr($str, 0, strlen($starter)) == $starter)
			return $starter;
	return null;
}

/** A wrapper to avoid an E_NOTICE warning when $s2 is unmatched in:
  * list($sl, $s2) = explode($c, $str, 2)
  */
function xsplit2($c, $str) {
	$ret = explode($c, $str, 2);
	if (!isset($ret[1])) $ret[1] = '';
	return $ret;
}

/** As for strpos but accepts an array of $needles rather than a single value
  * and returns a list where the first value is as for strpos and the second
  * is the matching $needles member. */
function xstrpos($haystack, $needles, $start = 0) {
	$ret = false;
	foreach ($needles as $needle) {
		$tmp = strpos($haystack, $needle, $start);
		if ($tmp !== false && ($tmp < $ret || $ret === false)) {
			$ret = $tmp;
			$char = $needle;
		}
	}
	return array($ret, $char);
}

/** Removes non-alphanumeric characters and returns the resulting string. */
function filter_non_alnum($str) {
	$ret = '';
	/* This is required to filter out non-ascii encoded characters - without
	 * it the treeview fails to expand e.g. Cyrillic titles through load-on-
	 * demand. */
	$str = urlencode($str);
	$len = strlen($str);
	for ($i=0; $i<$len; $i++) if (ctype_alnum($str[$i])) $ret .= $str[$i];
	return $ret;
}

/**
 * A wrapper around natsort() that provides zero-based sequential numeric keys
 * for the new order rather than retaining key-value associations as the base
 * function does.
 */
function natsort_newkeys(&$array) {
	natsort($array);
	$newarray = array();
	while ($x = array_shift($array)) $newarray[] = $x;
	$array = $newarray;
}

/**
 * Returns the index of the first entry in $array that is greater than or
 * equal to $val, or false if none are.  Assumes that $array has been sorted
 * using natsort_newkeys().
 */
function findfirstgte_natsort($array, $val) {
	$ret = false;
	$max = count($array);
	for ($i = 0; $i < $max; $i++) {
		if (compare_natsort($array[$i], $val) >= 0) {
			$ret = $i;
			break;
		}
	}
	return $ret;
}

/**
 * Compares $str1 against $str2 in natsort order, returning -1, 0 or 1 if
 * $str1 is less than, equal to or greater than $str2, respectively.
 */
function compare_natsort($str1, $str2) {
	if ($str1 == $str2) return 0;
	else {
		$array  = array($str1, $str2);
		natsort($array);
		reset($array);
		$first = each($array);
		return $first['value'] == $str1 ? -1 : 1;
	}
}

/** Returns a copy of $array for only those keys present in $keys. */
function x_array_key_values($array, $keys) {
	$ret = array();
	foreach (array_keys($array) as $k)
		if (in_array($k, $keys)) $ret[$k] = $array[$k];
	return $ret;
}

# support PHP4
if (!function_exists('array_intersect_key')) {
	function array_intersect_key($arr1, $arr2) {
		$ret = array();
		foreach ($arr1 as $k1 => $e1)
			if (array_key_exists($k1, $arr2))
				$ret[$k1] = $e1;
		return $ret;
	}
}

?>
