<?php

/**
 * SpecialHierarchy.php
 *
 * This file is part of the Hierarchy extension for MediaWiki; supporting the
 * Treeview skin.  Its code performs two main functions: ensuring cache-
 * consistency and providing server-side support for client-side load-on-demand
 * of treeview branches.
 *
 * Licenced under the General Public Licence 2, without warranty.
 *
 * @licence GPL2 http://www.gnu.org/copyleft/gpl.html
 * @author Laird Shaw <lairdshaw77@gmail.com>
 */

use MediaWiki\Title\Title;

$dir = dirname( __FILE__ ) . '/';
$wgExtensionMessagesFiles['Hierarchy'] = $dir.'Hierarchy.i18n.php';

use MediaWiki\MediaWikiServices;

global $IP; # not required for MW 1.5.x or 1.6.x
require_once(__DIR__.'/../Hierarchy.php');
/** for FauxRequestEx */
require_once(__DIR__.'/../State.php');
if (file_exists("$IP/includes/SpecialPage.php")) {
	require_once("$IP/includes/SpecialPage.php");
} else	require_once("$IP/includes/specialpage/SpecialPage.php");
/** for wfGetRedirectsTo() and wfGetSeeAlsos() */
require_once(__DIR__.'/../UtilityFunctions.php');
/** for TvQueryString::queryStrToArr() */
require_once(__DIR__.'/../QueryString.php');

/** This class implements a special page used by the Treeview skin to download
  * through javascript only the necessary nodes when expanding an unloaded part
  * of the treeview.
  * @package MediaWiki
  * @subpackage Extensions
  */
class SpecialHierarchy extends SpecialPage {
	/**#@+ @access public */
	function __construct() {
		parent::__construct('Hierarchy');

		global $wgCacheEpoch, $wgCacheEpochTouchFile, $wgUseFileCache,
		  $wgCachePages, $wgContLanguageCode, $wgSpecialPages;

		$tmp = @filemtime($wgCacheEpochTouchFile);
		if ($tmp !== false) {
			$lastTouched = gmdate('YmdHis', $tmp);
			$org = $wgCacheEpoch;
			$wgCacheEpoch = max($wgCacheEpoch, $lastTouched);
			if ($wgCacheEpoch != $org) {
				wfDebug(__METHOD__.": SET \$wgCacheEpoch to ".
				  "$wgCacheEpoch; was $org\n");
			}
		} else if (($wgUseFileCache || $wgCachePages) &&
		  !@touch($wgCacheEpochTouchFile)) {
			global $wgOut;
			$wgOut->showFatalError(wfMessage('tv_inaccessibletouch',
			  __METHOD__, '<code>'.$wgCacheEpochTouchFile.'</code>')
			  ->escaped());
		}
	}

	/** Main entry point function for SpecialHierarchy as a SpecialPage. */
	function execute($par) {
		/* Avoid $wgRequest because it includes cookies. */
		$queryParams = $_GET;
		unset($queryParams['title']);
		if (!$queryParams) {
			$this->showMainControls();
		} else	$this->handleLoadOnDemand_Xml();
	}

	function showMainControls() {
		global $wgOut, $wgHierarchyPage;
		$this->setHeaders();
		$wgOut->addHTML('<h1>Hierarchy/Treeview controls</h1>'."\n");
		list($tConfig, $fullMatch) = Hierarchy::getConfigPageTitle_();
		if (!$tConfig) {
			$wgOut->addHTML(wfMessage('tv_invconfpg',
			  $wgHierarchyPage)->escaped()."\n");
		} else {
			$sk = $this->getSkin();
			$confPage = $fullMatch? 'confpg': 'standinconfpg';
			$confTitle = $tConfig->getPrefixedText();
			$wgOut->addHTML('<dl>'."\n");
			foreach (array(
			  $confPage             => false,
			  'parsetreeview'       => true,
			  'rebuildtreeview'     => true,
			  'treeviewcachestatus' => true)
			as $key => $isAction) {
				if ($isAction && !RequestContext::getMain()->getUser()->isAllowed($key)) {
					continue;
				}
				$desc = wfMessage('tv_'.$key.'_desc')
				  ->escaped();
				$linkFunc = $isAction ?
				  'makeKnownLink' : 'makeLink';
				$linkRenderer = MediaWikiServices::getInstance()
				  ->getLinkRenderer();
				$link = $linkRenderer->$linkFunc($tConfig,
				  wfMessage('tv_'.$key, $confTitle)->escaped(),
				  [], $isAction ? ['action' => $key] : []);
				$wgOut->addHTML("<dt>$link</dt>\n".
				  "<dd>$desc</dd>\n");
			}
			$wgOut->addHTML('</dl>'."\n");
		}
	}

	/** Handles a load-on-demand request for a (set of) nodes.  Interprets
	  * the url query parameters, builds the branch(es) and outputs XML in
	  * place of MediaWiki's default output mechanisms.  The request is
	  * issued, and its result interpreted, by code in
	  * [mediawikiroot]/skins/treeview/treeview.js
	  */
	function handleLoadOnDemand_Xml() {
		global $wgOut, $wgInputEncoding, $wgRequest;
		/* Avoid E_NOTICE warnings during accesses to undefined array
		 * keys.*/
		// wfSuppressWarnings();

		$nodeIds = explode(',', $wgRequest->getText('expnodeids'));
		$base = $wgRequest->getVal('base');
		/* WebRequest::normalizeUnicode performs (all/most of?) the
		 * equivalent of urldecode. */
		$effFullUrl = /*urldecode(*/$wgRequest->getVal('effurl')/*)*/;
		if (empty($effFullUrl)) $effFullUrl = '';
		$tmp = strpos($effFullUrl, '?');
		$effQueryStr = $tmp!==false ? substr($effFullUrl, $tmp+1) : '';
		$fauxParams = TvQueryString::queryStrToArr($effQueryStr, false);
		if ($base && preg_match("#$base/(.*)#",$effFullUrl,$effTitle)) {
			$effTitle = $effTitle[1];
		} else {
			$effTitle = $wgRequest->getVal('effttl');
			/* When a url does not contain a title, e.g.
			 * index.php?action=edit, then MediaWiki uses the Main
			 * Page title. */
			if (!$effTitle) $effTitle = isset($fauxParams['title']) ?
			  $fauxParams['title'] : wfMessage('mainpage')->plain();
		}
		$tEffective = Title::newFromText($effTitle);
		$effWebRequest = new FauxRequestEx($fauxParams, $effFullUrl);
		$state = new TVstate(array(
			'tEffective' => $tEffective,
			'tRedirectInfo' => null, # unknown: follow on-demand
			'effectiveWebRequest' => $effWebRequest,
			'effectiveQueryStr' => $effQueryStr,
			'effectiveAction' => $effWebRequest->getVal('action')));
		$hierarchy = new Hierarchy($state);
		$skipStateRebuild = $wgRequest->getBool('skiplinkstaterebuild');
		wfDebug(__METHOD__.": retrieved from request :-\n".
		  "*effective title     : ".($tEffective
		    ? ('"'.$tEffective->getPrefixedText().'"')
		    : '[could not build Title from "'.$effTitle.'"]')."\n".
		  "*base                : ".($base ? "\"$base\"" : 'n/a')."\n".
		  "*expand-node ids     : ".var_export($nodeIds, true)."\n".
		  "*effective full url  : \"$effFullUrl\"\n".
		  "*effective query str : \"$effQueryStr\"\n".
		  '*skiplinkstaterebuild: "'.($skipStateRebuild?'1':'0')."\"\n".
		  "*decoded effective query string as:\n".
		    var_export($fauxParams, true)."\n"
		);

		$op = "<div>\n";
		$htmlArr = $hierarchy->buildHierarchyHtml($nodeIds,
		  $skipStateRebuild, '', true);
		foreach ($htmlArr as $nodeId => $html) {
			if ($html === false) $html = '<input type="hidden" '.
			  "class=\"notfound\">$nodeId</input>";
			$op .= "<div id=\"$nodeId\">$html</div>\n";
		}
		$expandeds = $hierarchy->getExpandedNodesStr();
		$op .= "<form action=\"#\">\n<input type=\"hidden\" ".
		  "id=\"expandednodes\" value=\"$expandeds\" />\n</form>\n";
		$op .= "</div>\n";

		$wgOut->disable();
		header('Content-type: text/xml; charset='.$wgInputEncoding);
//		wfDebug(__METHOD__.": returning as xml:\n$op\n");
		echo $op;

		// wfRestoreWarnings();
	}

	/**#@- */ # end public section
}

?>
